﻿;==================================================

;JEEAhk1FC

;==================================================

;AHK v2 functions for AHK v1 by jeeswg
;[first released: 2017-03-26]
;[updated: 2023-05-03]
;based on AHK v2.0.2
;tested with AHK v1.1.36

;use at your own risk,
;in particular, inspect and test the RegCreateKey/RegDelete/RegDeleteKey/RegWrite code for yourself

;==================================================

;KEY NOTES:

;note: versions of AHK v1.1:
;functions in this library should work correctly for the 'tested with AHK' version specified above,
;functions in this library are not guaranteed to work correctly for older AHK v1.1 versions,
;the earliest AHK version, where this library can run without a startup error, is AHK v1.1.29 (2018-05-25),
;before that, AHK v1.1 versions fail because of the presence of ObjGetBase,
;many individual functions can run in even earlier versions

;known issues/limitations:
;ClipboardAll: you can't create a class called ClipboardAll in AHK v1.1, so Buffer is used instead
;FileInstall: can only perform a FileCopy
;FileSelect: there may be a problem if trying to use FileSelect multiple times simultaneously
;InputBox: can only use * as a password character
;ListVars: can only list global variables
;Random: the function now supports Int64 values (not just Int32) but has not been extensively tested outside Int32
;Sort: WARNING: there may be a problem if trying to use Sort, with different callback functions, multiple times simultaneously
;StrPtr: if a literal string is passed, it is stored temporarily in an array (with slots reused after 1000 items)
;Type: doesn't fully match the AHK v2 function (it sometimes misidentifies a Float as a String, 'FloatOrString' could be introduced, a possible workaround: float += 0)
;further: in AHK v2 you need to use '&' to use 'VarRef' parameters (in AHK v1, you use no prefix to use 'ByRef' parameters)
;further: any instances of the word 'FIXME'
;further: FileAppend/MsgBox/InputBox run specific 'JEE_' functions if present

;additional functions:
;AHKFC_InputBoxSimple returns a string
;AHKFC_InputBoxObj returns an object

;==================================================

;FURTHER NOTES:

;force-expression style:
;parameters in lines containing commands and pseudo-operators (between (not used in this library)/contains/in/is),
;have all been converted to force-expression style (except for input/output variables, and numeric values),
;thus all literal strings are wrapped in double quotes,
;(if you don't write command parameters in this way, you need type information for each parameter in each command, to easily parse a line that starts with a command,)
;this facilitates parsing this script (manually/automatically), and doing (manual/automatic) string searches,
;it is particularly useful when parsing to check that variables in a function body, are defined as local/static/global variables, or as parameter variables (in the function header)

;note: functions that did exist in AHK v2, at some point, but were since removed from AHK v2,
;that have been maintained here:
;Input, InputEnd, RandomSeed, SoundGet, SoundSet, StringCaseSense

;for reference: AHK v1.1 functions which have had major changes in AHK v2:
;note: this library only replaces commands, not functions:
;DllCall: accepts objects with a Ptr property
;InStr/RegExMatch/RegExReplace/SubStr: negative StartPos parameter
;InStr: occurrence parameter
;NumPut/NumGet/StrPut/StrGet: accept objects with Ptr and Size properties (and File.RawRead, File.RawWrite)
;NumPut: parameters reordered
;RegExMatch: new default RegEx settings
;StrPut: now returns byte count, not character count
;WinActive/WinExist: WinTitle parameters accept objects with an Hwnd property

;a note on '0 is the same as specifying 0.5':
;this has been removed for: ClipWait, StatusBarWait, WinWait, WinWaitActive, WinWaitClose, WinWaitNotActive
;but is still in place for: WinClose, WinKill

;notes on WinTitle in functions:
;in most functions containing 'WinTitle', you find the first matching window,
;and do something to it, exceptions:
;GroupAdd
;WinActivateBottom [checks for the last match]
;WinGetCount
;WinGetIDLast [checks for the last match]['WinGetIDBottom']
;WinGetList
;WinWaitClose [can wait for multiple windows to close]
;WinWaitNotActive [waits for a non-match to be active]

;functions with Control/WinTitle parameters:
;Control and WinTitle (input) parameters: ControlXXX/EditXXX/ListViewGetContent/PostMessage/SendMessage
;WinTitle parameter only: ControlClick/ControlGetFocus/GroupAdd/MenuSelect/StatusBarXXX/WinXXX
;Control and WinTitle output parameters: MouseGetPos

;some custom auxiliary functions:
;Array (needed to modify '[]' array assignment), AHKFC_XXX, BIF_XXX, BIFEx_XXX
;note: AHKFC_XXX functions: might contain v1 code (only intended for this library)
;note: BIF_XXX/BIFEx_XXX functions: should be pure code that could work in v1 or v2 (intended as part of a stand-alone 'built-in functions' library)

;some custom class backports:
;Array, Buffer, Map, (plus A_TrayMenu.ClickCount)

;pseudo-functions:
;'MyClass.New(...)' (AHK v1) is just 'MyClass(...)' in AHK v2
;e.g. Array(), Map(), Object(), are not functions, but calls to class instance constructors

;==================================================

;SOME NOTES ON STATIC ASSIGNMENTS AND FUNCTION CALLS:

;some notes on 'static':
;in AHK v1: static assignments are executed at script startup
;in AHK v2: static assignments are executed the first time a 'static' line is encountered

;an example of static assignments in AHK v1:

;;EXAMPLE 1: doesn't work as hoped for (outputs a blank string):
;MyFunc1()
;{
;	static dummy := MyFunc2()
;}
;MyFunc2()
;{
;	static var := 123
;	MsgBox, % var ;(blank)
;}

;;EXAMPLE 2: does work as hoped for (outputs 123) (because MyFunc2 appears above MyFunc1):
;MyFunc2()
;{
;	static var := 123
;	MsgBox, % var ;(blank)
;}
;MyFunc1()
;{
;	static dummy := MyFunc2()
;}

;;EXAMPLE 3A: does work as hoped for (outputs 123) (safer, works regardless of the functions' positions):
;MyFunc1()
;{
;	static dummy := MyFunc2()
;}
;MyFunc2()
;{
;	static isready, var
;	if !VarSetCapacity(isready)
;	{
;		var := 123
;		isready := "1"
;	}
;	MsgBox, % var ;123
;}

;;EXAMPLE 3B: an improved version of Example 3A, when IsSet is available:
;MyFunc1()
;{
;	static dummy := MyFunc2()
;}
;MyFunc2()
;{
;	static isready, var
;	if !IsSet(isready)
;	{
;		var := 123
;		isready := 1
;	}
;	MsgBox, % var ;123
;}

;;EXAMPLE 3C: an even better version of Example 3A, when IsSet is available:
;MyFunc1()
;{
;	static dummy := MyFunc2()
;}
;MyFunc2()
;{
;	static var
;	if !IsSet(var)
;		var := 123
;	MsgBox, % var ;123
;}

;the explanation is as follows:
;in AHK v1:
;static assignments occur when the script loads,
;static assignments occur from top to bottom,
;in EXAMPLE 1, 'static dummy := MyFunc2()' calls MyFunc2,
;however, 'static var := 123' hasn't been executed yet,
;so var has no value,
;in EXAMPLE 3, we work around this by not doing any static assignments in MyFunc2,
;instead we use an 'isready' variable (which is declared as static, but not assigned a value),
;any assignments occur the first time the function is called,
;this principle is used in around half-a-dozen functions in this library,
;e.g. CallbackCreate/CallbackFree/ComCall/MsgBox/Type

;so, in short, we do this: 'static var',
;not this: 'static var := 1'

;note: we do 'isready := "1"', not 'isready := 1',
;because AHK v1's VarSetCapacity will only give a positive value for the *string* "1", and not the integer 1

;==================================================

;FOR REFERENCE: AHK v2 FUNCTIONS THAT HAVE VARREF ('BYREF') PARAMETERS:

;that aren't in this library:
;note: in AHK v2, enumerators use VarRef parameters
;(DllCall)
;IsSetRef
;LoadPicture
;RegExMatch
;RegExReplace
;StrReplace

;note: *NOT* IsSet (it does *not* use a VarRef parameter)

;that are in this library:
;(ComCall)
;CaretGetPos
;ControlGetPos
;FileGetShortcut
;ImageSearch
;MonitorGet
;MonitorGetWorkArea
;MouseGetPos
;PixelSearch
;Run
;RunWait
;SplitPath
;VarSetStrCapacity
;WinGetClientPos
;WinGetPos

;==================================================

;FOR REFERENCE: AHK v2 FUNCTIONS NOT IN THIS LIBRARY/CONTROL FLOW STATEMENTS:

;for reference: selected functions in AHK v1, not in AHK v2:
;(Func) [although a Func class exists in AHK v2]
;IsByRef
;IsFunc

;for reference: further functions in AHK v1, not in AHK v2:
;Asc/Exception/RegisterCallback/VarSetCapacity (replaced with Ord/Error/CallbackCreate/VarSetStrCapacity respectively)
;LV_XXX/SB_XXX/TV_XXX functions (handled by GUI objects)
;MenuGetHandle/MenuGetName (handled by Menu objects)
;some ComObjXXX/ObjXXX functions

;for reference: AHK v2 functions not recreated here:
;Gui
;GuiCtrlFromHwnd
;GuiFromHwnd
;InstallKeybdHook [key function]
;InstallMouseHook [key function]
;IsSetRef
;Menu
;MenuBar
;MenuFromHandle
;Persistent [key function]

;for reference: functions that exist both in AHK v1.1 and AHK v2 (so aren't recreated here):
;Abs
;ACos
;ASin
;ATan
;Ceil
;Chr
;ComObjActive
;ComObjArray
;ComObjConnect
;ComObject
;ComObjFlags
;ComObjGet
;ComObjQuery
;ComObjType
;ComObjValue
;Cos
;DllCall
;Exp
;FileExist
;FileOpen
;Floor
;Format
;GetKeyName
;GetKeySC
;GetKeyState
;GetKeyVK
;Hotstring
;IL_Add
;IL_Create
;IL_Destroy
;InputHook
;InStr
;IsLabel
;IsObject
;IsSet
;Ln
;LoadPicture
;Log
;LTrim
;Max
;Min
;Mod
;NumGet
;NumPut
;ObjAddRef
;ObjBindMethod
;Object
;ObjGetBase
;ObjGetCapacity
;ObjRelease
;ObjSetBase
;ObjSetCapacity
;OnClipboardChange
;OnError
;OnExit
;OnMessage
;Ord
;RegExMatch
;RegExReplace
;Round
;RTrim
;Sin
;Sqrt
;StrGet
;StrLen
;StrPut
;StrReplace
;StrSplit
;SubStr
;Tan
;Trim
;VerCompare
;WinActive
;WinExist

;for reference: control flow statements present in both AHK v1.1 and AHK v2:
;note: AHK v2 lacks Gosub
;Break
;Catch
;Continue
;Else
;Finally
;For
;Goto
;If
;Loop
;Return
;Switch
;Throw
;Try
;Until
;While

;==================================================

;LINKS:

;links:
;commands as functions (AHK v2 functions for AHK v1) - AutoHotkey Community
;https://autohotkey.com/boards/viewtopic.php?f=37&t=29689
;list of every command/function/variable from across all versions - AutoHotkey Community
;https://autohotkey.com/boards/viewtopic.php?f=74&t=27321&p=131642#p131642
;Alphabetical Command and Function Index | AutoHotkey
;https://www.autohotkey.com/docs/commands/
;Alphabetical Function Index | AutoHotkey v2
;https://lexikos.github.io/v2/docs/commands/index.htm

;see also (re. functions):
;GitHub - cocobelgica/AutoHotkey-Future: Port of AutoHotkey v2.0-a built-in functions for AHK v1.1+
;https://github.com/cocobelgica/AutoHotkey-Future
;AutoHotkey-Future/Lib at master · cocobelgica/AutoHotkey-Future · GitHub
;https://github.com/cocobelgica/AutoHotkey-Future/tree/master/Lib
;Default/Portable installation StdLib - AutoHotkey Community
;https://autohotkey.com/boards/viewtopic.php?f=13&t=10434

;see also (re. GUIs):
;objects: backport AHK v2 Gui/Menu classes to AHK v1 - AutoHotkey Community
;https://autohotkey.com/boards/viewtopic.php?f=37&t=43530

;==================================================

;INCOMPLETE FUNCTIONS:

;[use GUI library]
;Gui
;GuiCtrlFromHwnd
;GuiFromHwnd
;Menu
;MenuBar
;MenuFromHandle

InstallKeybdHook(Install:="", Force:="")
{
	throw Exception("Not implemented.", -1)
}
InstallMouseHook(Install:="", Force:="")
{
	throw Exception("Not implemented.", -1)
}
IsSetRef(ByRef Ref) ;ByRef, but function not implemented
{
	throw Exception("Not implemented.", -1)
}
Persistent(Persist:="")
{
	throw Exception("Not implemented.", -1)
}

GetMethod(Value, Name:="`f`a`b", ParamCount:="")
{
	if (Name = "`f`a`b")
	{
		if HasMethod(Value,, ParamCount)
			return Value
	}
	else if HasMethod(Value, Name, ParamCount)
		return Value[Name]
	throw Exception("", -1)
}
HasBase(Value, BaseObj)
{
	try
	{
		Loop
		{
			Value := ObjGetBase(Value)
			if !IsObject(Value)
				return False
			if (Value = BaseObj)
				return True
		}
	}
	return False
}
HasMethod(Value, Name:="`f`a`b", ParamCount:="")
{
	local IsNameOmitted, Method, MinParams, Type
	IsNameOmitted := (Name = "`f`a`b")
	Type := Type(Value)
	if (Type = "ComObject")
		throw Exception("", -1)
	if IsNameOmitted
	{
		if !IsObject(Value)
			return 0
		if (Type = "Func") ;note: Func has Call/IsVariadic/MaxParams/MinParams (and other) methods
		|| (Type = "BoundFunc") ;note: BoundFunc only has 'Call' method
			{} ;noop
		else
			throw Exception("", -1)
		Method := Value
	}
	else
	{
		try
			;Method := ObjRawGet(Value, Name) ;one level of depth only
			Method := Value[Name]
		catch
			return 0
		if !IsObject(Method)
			return 0
		Type := Type(Method)
	}
	MinParams := IsFunc(Method) - 1
	if (MinParams == -1)
		return 0
	if StrLen(ParamCount) && (ParamCount < MinParams)
		return 0
	if (Type = "Func")
	&& !Method.IsVariadic
	&& (ParamCount > Method.MaxParams)
		return 0
	return !!MinParams ;the AHK v2 function appears to return 1 or 0
}
HasProp(Value, Name)
{
	local Prop
	if Value.HasKey(Name)
		return 1
	try
	{
		Loop
		{
			Prop := ObjRawGet(Value, Name)
			if IsObject(Prop) || !!StrLen(Prop)
				return 1
			Value := ObjGetBase(Value)
			if !IsObject(Value)
				return 0
		}
	}
	return 0
}
ObjHasOwnProp(Obj, Name)
{
	return Obj.HasKey(Name)
}
ObjOwnPropCount(Obj)
{
	if Obj.HasKey("__Class") ;if a class object ('IsClassObj')
		throw Exception("Class object handling not implemented.", -1)
	return Obj.Count()
}
ObjOwnProps(Obj)
{
	return Obj
}

;==================================================

;FUNCTIONS:

BlockInput(OnOff)
{
	if OnOff in % "1,0"
		OnOff := OnOff ? "On" : "Off"
	BlockInput, % OnOff
}
Buffer(ByteCount, FillByte:="")
{
	return new Buffer(ByteCount, FillByte)
}
;CallbackCreate (see lower down)
;CallbackFree (see lower down)
CaretGetPos(ByRef OutputVarX:="", ByRef OutputVarY:="")
{
	local GUITHREADINFO, hWnd, hWndC, Mode, OriginX, OriginY, POINT, RECT, TID
	;this works but there was an issue regarding A_CaretX/A_CaretY not updating correctly:
	;OutputVarX := A_CaretX, OutputVarY := A_CaretY
	OutputVarX := OutputVarY := ""
	if !(hWnd := WinExist("A"))
		return 0 + 0
	VarSetCapacity(GUITHREADINFO, A_PtrSize=8?72:48, 0)
	NumPut(A_PtrSize=8?72:48, &GUITHREADINFO, 0, "UInt") ;cbSize
	TID := DllCall("user32\GetWindowThreadProcessId", "Ptr",hWnd, "Ptr",0, "UInt")
	if !DllCall("user32\GetGUIThreadInfo", "UInt",TID, "Ptr",&GUITHREADINFO)
		return 0 + 0
	hWndC := NumGet(&GUITHREADINFO, A_PtrSize=8?48:28, "Ptr") ;hwndCaret
	OutputVarX := NumGet(&GUITHREADINFO, A_PtrSize=8?56:32, "Int") ;rcCaret.left
	OutputVarY := NumGet(&GUITHREADINFO, A_PtrSize=8?60:36, "Int") ;rcCaret.top
	Mode := SubStr(A_CoordModeCaret, 1, 1)
	VarSetCapacity(POINT, 8)
	NumPut(OutputVarX, &POINT, 0, "Int")
	NumPut(OutputVarY, &POINT, 4, "Int")
	DllCall("user32\ClientToScreen", "Ptr",hWndC, "Ptr",&POINT)
	OutputVarX := NumGet(&POINT, 0, "Int")
	OutputVarY := NumGet(&POINT, 4, "Int")
	if (Mode = "S") ;screen
		return 0 + 1
	else if (Mode = "C") ;client
	{
		VarSetCapacity(POINT, 8, 0)
		DllCall("user32\ClientToScreen", "Ptr",hWnd, "Ptr",&POINT)
		OriginX := NumGet(&POINT, 0, "Int")
		OriginY := NumGet(&POINT, 4, "Int")
	}
	else if (Mode = "W") ;window
	{
		VarSetCapacity(RECT, 16, 0)
		DllCall("user32\GetWindowRect", "Ptr",hWnd, "Ptr",&RECT)
		OriginX := NumGet(&RECT, 0, "Int")
		OriginY := NumGet(&RECT, 4, "Int")
	}
	OutputVarX -= OriginX
	OutputVarY -= OriginY
	return 0 + 1
}
Click(Params*) ;Options
{
	local Args := "", i, Param
	for i, Param in Params
		Args .= " " Param
	Click, % Args
}
ClipboardAll(Data:="", Size:="")
{
	local BufAddr, BufSize, ClipSaved, oBuf
	if IsObject(Data)
	{
		BufAddr := Data.Ptr
		if BufAddr is not % "number"
			throw Exception("", -1)
		BufSize := Data.Size
		if BufSize is not % "number"
			throw Exception("", -1)
		if (Size == "")
			Size := Data.Size
		else if Size is not % "number"
			throw Exception("", -1) ;note: AHK v2 currently (unusually) treats non-numbers as 0
		oBuf := Buffer(Size)
		if (Size > Data.Size)
			Size := Data.Size
		DllCall("msvcrt\memmove", "Ptr",oBuf.Ptr, "Ptr",Data.Ptr, "UPtr",Size, "Cdecl Ptr")
		return oBuf
	}
	else if !StrLen(Data) && !StrLen(Size)
	{
		ClipSaved := ClipboardAll
		Size := StrLen(ClipSaved) << !!A_IsUnicode
		oBuf := Buffer(Size)
		DllCall("msvcrt\memmove", "Ptr",oBuf.Ptr, "Ptr",&ClipSaved, "UPtr",Size, "Cdecl Ptr")
		return oBuf
	}
	else if Data is % "integer"
	{
		if Size is not % "number"
			throw Exception("", -1)
		oBuf := Buffer(Size)
		DllCall("msvcrt\memmove", "Ptr",oBuf.Ptr, "Ptr",Data, "UPtr",Size, "Cdecl Ptr")
		return oBuf
	}
	throw Exception("", -1)
}
ClipWait(Timeout:="", WaitForAnyData:=0)
{
	if (Timeout == 0)
		Timeout := 0.001
	ClipWait, % Timeout, % WaitForAnyData
	return !ErrorLevel
}
;ComCall (see lower down)
ComObjFromPtr(DispPtr)
{
	return ComObject(9, DispPtr, 1)
}
ComValue(VarType, Value, Flags:=0)
{
	return ComObjParameter(VarType, Value, Flags)
}
ControlAddItem(String, Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local CtlClass, hCtl
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	Control, % "Add", % String,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	WinGetClass, CtlClass, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	if InStr(CtlClass, "List")
		SendMessage, 0x18B, 0, -1,, % "ahk_id " hCtl ;LB_GETCOUNT := 0x18B
	else if InStr(CtlClass, "Combo")
		SendMessage, 0x146, 0, -1,, % "ahk_id " hCtl ;CB_GETCOUNT := 0x146
	if (ErrorLevel = "FAIL")
		throw Exception("", -1)
	return 0 + ErrorLevel
}
ControlChooseIndex(N, Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local CtlClass, hCtl
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	WinGetClass, CtlClass, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	if InStr(CtlClass, "Tab")
	{
		SendMessage, 0x1330, % N-1, 0,, % "ahk_id " hCtl ;TCM_SETCURFOCUS := 0x1330
		if (ErrorLevel = "FAIL")
			throw Exception("", -1)
		Sleep, 0
		SendMessage, 0x130C, % N-1, 0,, % "ahk_id " hCtl ;TCM_SETCURSEL := 0x130C
		if (ErrorLevel = "FAIL")
			throw Exception("", -1)
	}
	else if (N != 0)
	{
		Control, % "Choose", % N,, % "ahk_id " hCtl
		if ErrorLevel
			throw Exception("", -1)
	}
	else if InStr(CtlClass, "List")
	{
		SendMessage, 0x185, 0, -1,, % "ahk_id " hCtl ;LB_SETSEL := 0x185
		if (ErrorLevel = "FAIL")
			throw Exception("", -1)
	}
	else if InStr(CtlClass, "Combo")
	{
		SendMessage, 0x14E, 0, -1,, % "ahk_id " hCtl ;CB_SETCURSEL := 0x14E
		if (ErrorLevel = "FAIL")
			throw Exception("", -1)
	}
	else
		throw Exception("", -1)
}
ControlChooseString(String, Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local CtlClass, hCtl
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	Control, % "ChooseString", % String,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	WinGetClass, CtlClass, % "ahk_id " hCtl
	if InStr(CtlClass, "List")
		SendMessage, 0x188, 0, -1,, % "ahk_id " hCtl ;LB_GETCURSEL := 0x188
	else if InStr(CtlClass, "Combo")
		SendMessage, 0x147, 0, -1,, % "ahk_id " hCtl ;CB_GETCURSEL := 0x147
	if (ErrorLevel = "FAIL")
		throw Exception("", -1)
	return 1 + ErrorLevel
}
ControlClick(ControlOrPos:="", WinTitle:="", WinText:="", WhichButton:="", ClickCount:="", Options:="", ExcludeTitle:="", ExcludeText:="")
{
	local CWinX, CWinY, hCtl := 0, hWnd, Match, WinX, WinY
	if ControlOrPos is % "integer"
	{
		if (Type(ControlOrPos) == "Integer")
			hCtl := WinExist("ahk_id " ControlOrPos)
	}
	if IsObject(ControlOrPos)
	{
		if !ControlOrPos.Hwnd
			throw Exception("", -1)
		hCtl := WinExist("ahk_id " ControlOrPos.Hwnd)
	}
	if !hCtl
		hWnd := AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
	&& RegExMatch(ControlOrPos, "Oi)X(\d+)[ `t]+Y(\d+)", Match)
	{
		WinGetPos, WinX, WinY,,, % "ahk_id " hWnd
		if ErrorLevel
			throw Exception("", -1)
		try WinGetClientPos(CWinX, CWinY,,, "ahk_id " hWnd)
		catch
			throw Exception("", -1)
		ControlOrPos := "X" (Match[1]+CWinX-WinX) " Y" (Match[2]+CWinY-WinY)
	}
	if hCtl
		ControlClick, % ControlOrPos, % "ahk_id " hCtl,, % WhichButton, % ClickCount, % Options
	else if hWnd
		ControlClick, % ControlOrPos, % "ahk_id " hWnd,, % WhichButton, % ClickCount, % Options
	else
		ErrorLevel := 1
	if ErrorLevel
		throw Exception("", -1)
}
;how ControlClick would look if a Control parameter added (untested):
;ControlClickAlt(Pos:="", Control:="", WinTitle:="", WinText:="", WhichButton:="", ClickCount:="", Options:="", ExcludeTitle:="", ExcludeText:="")
;{
;	local CWinX, CWinY, hCtl := 0, hWnd, Match, WinX, WinY
;	hWnd := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
;	if !hWnd
;		throw Exception("", -1)
;
;	if StrLen(Pos)
;	&& RegExMatch(ControlOrPos, "Oi)X(\d+)[ `t]+Y(\d+)", Match)
;	{
;		WinGetPos, WinX, WinY,,, % "ahk_id " hWnd
;		if ErrorLevel
;			throw Exception("", -1)
;		try WinGetClientPos(CWinX, CWinY,,, "ahk_id " hWnd)
;		catch
;			throw Exception("", -1)
;		ControlOrPos := "X" (Match[1]+CWinX-WinX) " Y" (Match[2]+CWinY-WinY)
;	}
;
;	ControlClick, % Pos, % "ahk_id " hWnd,, % WhichButton, % ClickCount, % Options
;	if ErrorLevel
;		throw Exception("", -1)
;}
ControlDeleteItem(N, Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	Control, % "Delete", % N,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
}
ControlExist(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl
	if Control is % "integer"
	{
		if (Type(Control) == "Integer")
			return 0 + WinExist("ahk_id " Control)
	}
	if IsObject(Control)
	{
		if !Control.Hwnd
			throw Exception("", -1)
		return 0 + WinExist("ahk_id " Control.Hwnd)
	}
	if (Control == "")
		return 0 + AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
	ControlGet, hCtl, % "Hwnd",, % Control, % "ahk_id " AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
	if ErrorLevel
		return 0 + 0
	return 0 + hCtl
}
ControlFindItem(String, Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl, OutputVar
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlGet, OutputVar, % "FindString", % String,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	;note: AHK v2: warning: unintuitive: if no match is found, an exception is thrown
	return 0 + OutputVar
}
ControlFocus(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlFocus,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
}
ControlGetChecked(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl, OutputVar
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlGet, OutputVar, % "Checked",,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	return 0 + OutputVar
}
ControlGetChoice(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl, OutputVar
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlGet, OutputVar, % "Choice",,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	return OutputVar
}
ControlGetClassNN(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local CtlClass, hCtl, hCtl2, hWnd
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	WinGetClass, CtlClass, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	;hWnd := DllCall("user32\GetAncestor", "Ptr",hCtl, "UInt",1, "Ptr") ;GA_PARENT := 1
	;hWnd := DllCall("user32\GetAncestor", "Ptr",hCtl, "UInt",2, "Ptr") ;GA_ROOT := 2
	hWnd := AHKFC_GetNonChildParent(hCtl) ;non-child parent
	Loop
	{
		ControlGet, hCtl2, % "Hwnd",, % CtlClass A_Index, % "ahk_id " hWnd
		if !hCtl2
			break
		else if (hCtl == hCtl2)
			return CtlClass A_Index
	}
}
ControlGetEnabled(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl, OutputVar
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlGet, OutputVar, % "Enabled",,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	return 0 + OutputVar
}
ControlGetExStyle(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl, OutputVar
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlGet, OutputVar, % "ExStyle",,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	return 0 + OutputVar
}
ControlGetFocus(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local ClassNN, hCtl, hWnd
	hWnd := AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hWnd
		throw Exception("Target window not found.", -1)
	ControlGetFocus, ClassNN, % "ahk_id " hWnd
	if ErrorLevel
		throw Exception("", -1)
	ControlGet, hCtl, % "Hwnd",, % ClassNN, % "ahk_id " hWnd
	if ErrorLevel
		throw Exception("", -1)
	return 0 + hCtl
}
ControlGetHwnd(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl
	if Control is % "integer"
	{
		if (Type(Control) == "Integer")
			return 0 + WinExist("ahk_id " Control)
	}
	if IsObject(Control)
	{
		if !Control.Hwnd
			throw Exception("", -1)
		return 0 + WinExist("ahk_id " Control.Hwnd)
	}
	if (Control == "")
		return 0 + AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
	ControlGet, hCtl, % "Hwnd",, % Control, % "ahk_id " AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
	if ErrorLevel
		throw Exception("", -1)
	return 0 + hCtl
}
ControlGetIndex(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local CtlClass, hCtl, Msg
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	WinGetClass, CtlClass, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	if InStr(CtlClass, "Combo")
		Msg := 0x147 ;CB_GETCURSEL := 0x147
	else if InStr(CtlClass, "List")
		Msg := 0x188 ;LB_GETCURSEL := 0x188
	else if InStr(CtlClass, "Tab")
		Msg := 0x130B ;TCM_GETCURSEL := 0x130B
	else
		throw Exception("", -1)
	SendMessage, % Msg, 0, 0,, % "ahk_id " hCtl,,,, 2000
	if (ErrorLevel = "FAIL")
		throw Exception("", -1)
	return (ErrorLevel << 32 >> 32) + 1

	;alternative code for Tab controls:
	;ControlGet, OutputVar, % "Tab",,, % "ahk_id " hCtl
}
ControlGetItems(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl, OutputVar
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlGet, OutputVar, % "List",,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	return StrSplit(OutputVar, "`n")
}
ControlGetPos(ByRef X:="", ByRef Y:="", ByRef Width:="", ByRef Height:="", Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)

	;client coordinates:
	BIF_ControlGetPos(X, Y, Width, Height, hCtl)

	;if ErrorLevel
	;	throw Exception("", -1)
	;note: no return value, returns values via the ByRef variables
}
ControlGetStyle(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl, OutputVar
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlGet, OutputVar, % "Style",,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	return 0 + OutputVar
}
ControlGetText(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl, OutputVar
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlGetText, OutputVar,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	return OutputVar
}
ControlGetVisible(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl, OutputVar
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlGet, OutputVar, % "Visible",,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	return 0 + OutputVar
}
ControlHide(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	Control, % "Hide",,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
}
ControlHideDropDown(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	Control, % "HideDropDown",,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
}
ControlMove(X:="", Y:="", Width:="", Height:="", Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)

	;client coordinates:
	BIF_ControlMove(X, Y, Width, Height, hCtl)

	;if ErrorLevel
	;	throw Exception("", -1)
}
ControlSend(Keys, Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlSend, % "ahk_parent", % Keys, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
}
ControlSendText(Keys, Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	if (StrSplit(A_AhkVersion, ".")[3] >= 27) ;[v1.1.27+]
		ControlSend, % "ahk_parent", % "{Text}" Keys, % "ahk_id " hCtl
	else
	{
		Keys := StrReplace(Keys, "`r`n", "`n")
		ControlSend, % "ahk_parent", % "{Raw}" Keys, % "ahk_id " hCtl
	}
	if ErrorLevel
		throw Exception("", -1)
}
ControlSetChecked(Value, Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local Boolean, hCtl
	if Value in % "On,Off,Toggle"
		throw Exception("Parameter #1 is invalid.", -1)
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlGet, Boolean, % "Checked",,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	if (Value == -1)
		Value := !Boolean
	if (Value == 1)
		Control, % "Check",,, % "ahk_id " hCtl
	else if (Value == 0)
		Control, % "Uncheck",,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
}
ControlSetEnabled(Value, Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local Boolean, hCtl
	if Value in % "On,Off,Toggle"
		throw Exception("Parameter #1 is invalid.", -1)
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlGet, Boolean, % "Enabled",,, % "ahk_id " hCtl
	if (Value == -1)
		Value := !Boolean
	if (Value == 1)
		Control, % "Enable",,, % "ahk_id " hCtl
	else if (Value == 0)
		Control, % "Disable",,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
}
ControlSetExStyle(Value, Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	Control, % "ExStyle", % Value,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
}
ControlSetStyle(Value, Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	Control, % "Style", % Value,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
}
ControlSetText(ByRef NewText, Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="") ;ByRef for performance
{
	local hCtl
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlSetText,, % NewText, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
}
ControlShow(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	Control, % "Show",,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
}
ControlShowDropDown(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	Control, % "ShowDropDown",,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
}
CoordMode(TargetType, RelativeTo:="Screen")
{
	local OldValue
	if RelativeTo not in % ",Client,Screen,Window"
		throw Exception("Invalid value.`n`n" "Specifically: " RelativeTo, -1)
	;if TargetType not in % "Caret,Menu,Mouse,Pixel,ToolTip"
	;	throw Exception("Invalid value.`n`n" "Specifically: " TargetType, -1)
	OldValue := A_CoordMode%TargetType%
	CoordMode, % TargetType, % RelativeTo
	return OldValue
}
Critical(OnOffNumeric:="")
{
	local OldValue := A_IsCritical
	Critical, % OnOffNumeric
	return OldValue
}
DateAdd(DateTime, Time, TimeUnits)
{
	if StrLen(DateTime)
	{
		if DateTime is not % "time"
			throw Exception("Parameter #1 is invalid.", -1)
		if (StrLen(DateTime) & 1) || (StrLen(DateTime) > 14)
			throw Exception("Parameter #1 is invalid.", -1)
	}
	;[FIXME] AHK v2 DateTime parameters: unfortunately DateAdd doesn't currently accept a blank string, this backport does, DateDiff does (wish list idea)
	;else
	;	throw Exception("Parameter #1 is invalid.", -1)
	if (TimeUnits == "")
		throw Exception("Parameter #3 is invalid.", -1)
	EnvAdd, DateTime, % Time, % TimeUnits
	return 0 + DateTime
}
DateDiff(DateTime1, DateTime2, TimeUnits)
{
	if StrLen(DateTime1)
	{
		if DateTime1 is not % "time"
			throw Exception("Parameter #1 is invalid.", -1)
		if (StrLen(DateTime1) & 1) || (StrLen(DateTime1) > 14)
			throw Exception("Parameter #1 is invalid.", -1)
	}
	if StrLen(DateTime2)
	{
		if DateTime2 is not % "time"
			throw Exception("Parameter #2 is invalid.", -1)
		if (StrLen(DateTime2) & 1) || (StrLen(DateTime2) > 14)
			throw Exception("Parameter #2 is invalid.", -1)
	}
	if (TimeUnits == "")
		throw Exception("Parameter #3 is invalid.", -1)
	EnvSub, DateTime1, % DateTime2, % TimeUnits
	return 0 + DateTime1
}
DetectHiddenText(Mode)
{
	local OldValue := A_DetectHiddenText
	;note: AHK v2: On/Off not listed, but don't throw
	;if Mode in % "On,Off"
	;	throw Exception("Parameter #1 is invalid.", -1)
	if Mode in % "1,0"
		Mode := Mode ? "On" : "Off"
	DetectHiddenText, % Mode
	return OldValue
}
DetectHiddenWindows(Mode)
{
	local OldValue := A_DetectHiddenWindows
	;note: AHK v2: On/Off not listed, but don't throw
	;if Mode in % "On,Off"
	;	throw Exception("Parameter #1 is invalid.", -1)
	if Mode in % "1,0"
		Mode := Mode ? "On" : "Off"
	DetectHiddenWindows, % Mode
	return OldValue
}
DirCopy(Source, Dest, Overwrite:=0)
{
	FileCopyDir, % Source, % Dest, % Overwrite
	if ErrorLevel
		throw Exception("", -1)
}
DirCreate(DirName)
{
	FileCreateDir, % DirName
	if ErrorLevel
		throw Exception("", -1)
}
DirDelete(DirName, Recurse:=0)
{
	FileRemoveDir, % DirName, % Recurse
	if ErrorLevel
		throw Exception("", -1)
}
DirExist(FilePattern)
{
	local Attrib
	Loop Files, % FilePattern, % "D"
		return A_LoopFileAttrib ;note: if a dir is found, this will contain 'D' and possibly other characters
	;handle drives:
	Attrib := FileExist(FilePattern)
	return InStr(Attrib, "D") ? Attrib : ""
}
DirMove(Source, Dest, Flag:=0)
{
	FileMoveDir, % Source, % Dest, % Flag
	if ErrorLevel
		throw Exception("", -1)
}
DirSelect(StartingFolder:="", Options:=1, Prompt:="")
{
	local OutputVar
	FileSelectFolder, OutputVar, % StartingFolder, % Options, % Prompt
	;note: AHK v2: does not throw
	if ErrorLevel
		return ""
	return OutputVar
}
Download(URL, Filename)
{
	UrlDownloadToFile, % URL, % Filename
	if ErrorLevel
		throw Exception("Failed", -1)
}
DriveEject(Drive:="")
{
	Drive, % "Eject", % Drive
	if ErrorLevel
		throw Exception("", -1)
}
DriveGetCapacity(Path)
{
	local OutputVar
	DriveGet, OutputVar, % "Capacity", % Path
	if ErrorLevel
		throw Exception("", -1)
	return 0 + OutputVar
}
DriveGetFileSystem(Drive)
{
	local OutputVar
	DriveGet, OutputVar, % "FileSystem", % Drive
	if ErrorLevel
		throw Exception("", -1)
	return OutputVar
}
DriveGetLabel(Drive)
{
	local OutputVar
	DriveGet, OutputVar, % "Label", % Drive
	if ErrorLevel
		throw Exception("", -1)
	return OutputVar
}
DriveGetList(Type:="")
{
	local OutputVar
	DriveGet, OutputVar, % "List", % Type
	if ErrorLevel
		throw Exception("", -1)
	return OutputVar
}
DriveGetSerial(Drive)
{
	local OutputVar
	DriveGet, OutputVar, % "Serial", % Drive
	if ErrorLevel
		throw Exception("", -1)
	return OutputVar
}
DriveGetSpaceFree(Path)
{
	local OutputVar
	DriveSpaceFree, OutputVar, % Path
	if ErrorLevel
		throw Exception("", -1)
	return 0 + OutputVar
}
DriveGetStatus(Path)
{
	local OutputVar
	DriveGet, OutputVar, % "Status", % Path
	if ErrorLevel
		throw Exception("", -1)
	return OutputVar
}
DriveGetStatusCD(Drive:="")
{
	local OutputVar
	DriveGet, OutputVar, % "StatusCD", % Drive
	if ErrorLevel
		throw Exception("", -1)
	return OutputVar
}
DriveGetType(Path)
{
	local OutputVar
	DriveGet, OutputVar, % "Type", % Path
	if ErrorLevel
		throw Exception("", -1)
	return OutputVar
}
DriveLock(Drive)
{
	Drive, % "Lock", % Drive
	if ErrorLevel
		throw Exception("", -1)
}
DriveRetract(Drive:="")
{
	Drive, % "Eject", % Drive, 1
	if ErrorLevel
		throw Exception("", -1)
}
DriveSetLabel(Drive, NewLabel:="")
{
	Drive, % "Label", % Drive, % NewLabel
	if ErrorLevel
		throw Exception("", -1)
}
DriveUnlock(Drive)
{
	Drive, % "Unlock", % Drive
	if ErrorLevel
		throw Exception("", -1)
}
Edit()
{
	Edit
}
EditGetCurrentCol(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl, OutputVar
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlGet, OutputVar, % "CurrentCol",,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	return 0 + OutputVar
}
EditGetCurrentLine(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl, OutputVar
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlGet, OutputVar, % "CurrentLine",,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	return 0 + OutputVar
}
EditGetLine(N, Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl, OutputVar, Pos
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlGet, OutputVar, % "Line", % N,, % "ahk_id " hCtl

	;to counteract blank lines being considered an error:
	;'If the specified line number is blank ..., ErrorLevel is set to 1 and OutputVar is made blank.'
	;note: fixed in AHK v1.1.33
	if ErrorLevel
	{
		;note: if EM_LINEINDEX does not return -1, and EM_LINELENGTH returns 0, then the line is a valid (blank) line
		SendMessage, 0xBB, % N-1, 0,, % "ahk_id " hCtl ;EM_LINEINDEX := 0xBB ;line index (0-based) to char index (line's first char) (0-based)
		Pos := ErrorLevel
		if (Pos != "FAIL") && (Pos != 0xFFFFFFFF) ;-1
			SendMessage, 0xC1, % Pos, 0,, % "ahk_id " hCtl ;EM_LINELENGTH := 0xC1 ;char index (0-based) to length of line containing char
		else
			ErrorLevel := 1
	}

	if ErrorLevel
		throw Exception("", -1)
	return OutputVar
}
EditGetLineCount(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl, OutputVar
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlGet, OutputVar, % "LineCount",,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	return 0 + OutputVar
}
EditGetSelectedText(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl, OutputVar
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlGet, OutputVar, % "Selected",,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	return OutputVar
}
EditPaste(ByRef String, Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="") ;ByRef for performance
{
	local hCtl
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	Control, % "EditPaste", % String,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
}
EnvGet(EnvVarName)
{
	local OutputVar
	EnvGet, OutputVar, % EnvVarName
	return OutputVar
}
EnvSet(EnvVar, Value:="`f`a`b")
{
	if (Value = "`f`a`b") ;if value omitted, delete EnvVar
		ErrorLevel := !DllCall("kernel32\SetEnvironmentVariable", "Str",EnvVar, "Ptr",0)
	else
		EnvSet, % EnvVar, % Value
	if ErrorLevel
		throw Exception("", -1)
}
Error(Message, Params*) ;Message, What, Extra
{
	local What
	if (Params.Length() > 2)
		throw Exception("Too many parameters passed to function.", -1)
	if !Params.HasKey(1)
		Params[1] := 0
	What := Params[1]
	if What is % "integer"
		Params[1]--
	return Exception(Message, Params*)
}
Exit(ExitCode:=0)
{
	Exit, % ExitCode
}
ExitApp(ExitCode:=0)
{
	ExitApp, % ExitCode
}
;FileAppend(ByRef Text, Filename, Options) ;custom variant: requires that all 3 parameters be specified ;ByRef for performance
FileAppend(ByRef Text, Filename:="", Options:="") ;ByRef for performance
{
	local BufAddr, BufSize, Encoding, EOL, oFile
	Encoding := A_FileEncoding
	EOL := "*"
	Loop Parse, Options, % " `t"
	{
		if (A_LoopField = "`n")
			EOL := ""
		else if RegExMatch(A_LoopField, "i)^(UTF-|CP)")
			Encoding := A_LoopField
	}

	;warn if no encoding specified:
	;warn if appending non-ASCII to a file without a BOM:
	;warn if encoding mismatch (UTF-8 v. UTF-16):
	;log when files are created:
	static FuncLogSafety := Func("JEE_FileAppendLogAndSafetyCheck")
	if IsObject(FuncLogSafety)
		%FuncLogSafety%(Filename, Text, Encoding)

	;handle binary data:
	if IsObject(Text)
	{
		BufAddr := Text.Ptr
		if BufAddr is not % "number"
			throw Exception("", -1)
		BufSize := Text.Size
		if BufSize is not % "number"
			throw Exception("", -1)
		ErrorLevel := 0
		if (Encoding = "UTF-8") || (Encoding = "UTF-16")
			FileAppend, % "", % Filename, % Encoding
		else if !FileExist(Filename)
			FileAppend, % "", % Filename, % "CP1252"
		if ErrorLevel || !FileExist(Filename)
		|| !(oFile := FileOpen(Filename, "a"))
			throw Exception("", -1)
		oFile.RawWrite(Text.Ptr, Text.Size)
		oFile.Close()
		return
	}

	FileAppend, % Text, % EOL Filename, % Encoding
	if ErrorLevel
		throw Exception("", -1)
}
FileCopy(SourcePattern, DestPattern, Overwrite:=0)
{
	if !RegExMatch(SourcePattern, "[?*]") && !FileExist(SourcePattern)
		throw Exception("", -1)
	FileCopy, % SourcePattern, % DestPattern, % Overwrite
	if ErrorLevel
		throw Exception("", -1, ErrorLevel) ;note: Exception.Extra is set to the number of failures
}
FileCreateShortcut(Target, LinkFile, WorkingDir:="", Args:="", Description:="", IconFile:="", ShortcutKey:="", IconNumber:="", RunState:=1)
{
	FileCreateShortcut, % Target, % LinkFile, % WorkingDir, % Args, % Description, % IconFile, % ShortcutKey, % IconNumber, % RunState
	if ErrorLevel
		throw Exception("", -1)
}
FileDelete(FilePattern)
{
	FileDelete, % FilePattern
	if ErrorLevel
		throw Exception("", -1, ErrorLevel) ;note: Exception.Extra is set to the number of failures
}
FileEncoding(Encoding)
{
	local OldValue := A_FileEncoding
	if (OldValue == "")
		OldValue := "CP0"
	if (Encoding == "")
		Encoding := "CP0"
	else if Encoding is % "digit"
		Encoding := "CP" Encoding
	FileEncoding, % Encoding
	return OldValue
}
FileGetAttrib(Filename:="")
{
	local OutputVar
	FileGetAttrib, OutputVar, % Filename
	if ErrorLevel
		throw Exception("", -1)
	return OutputVar
}
FileGetShortcut(LinkFile, ByRef OutTarget:="", ByRef OutDir:="", ByRef OutArgs:="", ByRef OutDescription:="", ByRef OutIcon:="", ByRef OutIconNum:="", ByRef OutRunState:="")
{
	FileGetShortcut, % LinkFile, OutTarget, OutDir, OutArgs, OutDescription, OutIcon, OutIconNum, OutRunState
	if ErrorLevel
		throw Exception("", -1)
	;note: no return value, returns values via the ByRef variables
}
FileGetSize(Filename:="", Units:="")
{
	local OutputVar
	if !FileExist(Filename)
		throw Exception("The system cannot find the file specified.", -1)
	FileGetSize, OutputVar, % Filename, % Units
	if ErrorLevel
		throw Exception("", -1)
	return 0 + OutputVar
}
FileGetTime(Filename:="", WhichTime:="M")
{
	local OutputVar
	if !FileExist(Filename)
		throw Exception("The system cannot find the file specified.", -1)
	FileGetTime, OutputVar, % Filename, % WhichTime
	if ErrorLevel
		throw Exception("", -1)
	;[FIXME] AHK v2 returns a string, an integer might be more logical (wish list idea)
	return 0 + OutputVar
}
FileGetVersion(Filename:="")
{
	local OutputVar
	FileGetVersion, OutputVar, % Filename
	if ErrorLevel
		throw Exception("", -1)
	return OutputVar
}
FileInstall(Source, Dest, Overwrite:=0)
{
	;If this function is used in a normal (uncompiled) script, a simple file copy will be performed instead
	FileCopy, % Source, % Dest, % Overwrite
	if ErrorLevel
		throw Exception("", -1)
}
FileMove(SourcePattern, DestPattern, Overwrite:=0)
{
	if !RegExMatch(SourcePattern, "[?*]") && !FileExist(SourcePattern)
		throw Exception("", -1)
	if InStr(FileExist(SourcePattern), "D") ;[FIXME] probably always undesirable, should throw (wish list idea)
		MsgBox, % "custom warning: FileMove will attempt to move a folder"
	FileMove, % SourcePattern, % DestPattern, % Overwrite
	if ErrorLevel
		throw Exception("", -1, ErrorLevel) ;note: Exception.Extra is set to the number of failures
}
FileRead(Filename, Options:="")
{
	local BytesRead, oBuf, oFile, Options2 := "", OutputVar, Raw := 0, Size := ""
	Loop Parse, Options, % " `t"
	{
		if (SubStr(A_LoopField, 1, 1) = "m")
		{
			Options2 .= "*" A_LoopField " "
			Size := SubStr(A_LoopField, 2)
		}
		else if (A_LoopField = "`n")
			Options2 .= "*t "
		else if (SubStr(A_LoopField, 1, 2) = "CP")
			Options2 .= "*" SubStr(A_LoopField, 2) " "
		else if (SubStr(A_LoopField, 1, 5) = "UTF-8")
			Options2 .= "*P65001 "
		else if (SubStr(A_LoopField, 1, 6) = "UTF-16")
			Options2 .= "*P1200 "
		else if (A_LoopField = "RAW")
			Raw := 1
	}
	if Raw
	{
		if !(oFile := FileOpen(Filename, "r"))
			return ""
		;Size := (Size == "") ? oFile.Length : Min(Size, oFile.Length)
		Size := (Size == "") ? oFile.Length : (Size < oFile.Length ? Size : oFile.Length)
		oBuf := Buffer(Size)
		if (Size == 0)
		{
			oFile.Close()
			return oBuf
		}
		oFile.Pos := 0
		BytesRead := oFile.RawRead(oBuf.Ptr, Size)
		oFile.Close()
		if (BytesRead != Size)
			return ""
		return oBuf
	}
	FileRead, OutputVar, % Options2 Filename
	if ErrorLevel
		throw Exception("Filename: " Filename, -1) ;[FIXME] AHK v2 message doesn't show path (wish list idea)
	return OutputVar
}
FileRecycle(FilePattern)
{
	FileRecycle, % FilePattern
	if ErrorLevel
		throw Exception("", -1)
}
FileRecycleEmpty(DriveLetter:="")
{
	FileRecycleEmpty, % DriveLetter
	if ErrorLevel
		throw Exception("", -1)
}
FileSelect(Options:=0, RootDir_Filename:="", Title:="", Filter:="")
{
	local OSVersion, OutputVar
	OSVersion := DllCall("kernel32\GetVersion", "UInt")
	if (OSVersion < 6) ;IFileDialog requires Windows Vista or later
	{
		;note: AHK v2: FileSelect does not throw
		if InStr(Options, "D") ;fall back to the simple DirSelect
		{
			FileSelectFolder, OutputVar, % RootDir_Filename, 0x3, % Title
			if ErrorLevel
				return ""
		}
		else
			FileSelectFile, OutputVar, % Options, % RootDir_Filename, % Title, % Filter
		return OutputVar
	}
	return BIF_FileSelect(Options, RootDir_Filename, Title, Filter)
}
FileSetAttrib(Attributes, FilePattern:="", Mode:="")
{
	if !RegExMatch(Attributes, "^[+\-\^]")
	{
		FileSetAttrib, % "-RASHOT", % FilePattern, % InStr(Mode, "D") ? (InStr(Mode, "F") ? 1 : 2) : 0, % !!InStr(Mode, "R")
		Attributes := "+" Attributes
	}
	FileSetAttrib, % Attributes, % FilePattern, % InStr(Mode, "D") ? (InStr(Mode, "F") ? 1 : 2) : 0, % !!InStr(Mode, "R")
	if ErrorLevel
		throw Exception("", -1, ErrorLevel) ;note: Exception.Extra is set to the number of failures
}
FileSetTime(YYYYMMDDHH24MISS:="", FilePattern:="", WhichTime:="M", Mode:="")
{
	if StrLen(YYYYMMDDHH24MISS)
	{
		if YYYYMMDDHH24MISS is not % "time"
			throw Exception("Parameter #1 is invalid.`n`n" "Specifically: " YYYYMMDDHH24MISS, -1)
		if (StrLen(YYYYMMDDHH24MISS) & 1) || (StrLen(YYYYMMDDHH24MISS) > 14)
			throw Exception("Parameter #1 is invalid.`n`n" "Specifically: " YYYYMMDDHH24MISS, -1)
	}
	FileSetTime, % YYYYMMDDHH24MISS, % FilePattern, % WhichTime, % InStr(Mode, "D") ? (InStr(Mode, "F") ? 1 : 2) : 0, % !!InStr(Mode, "R")
	if ErrorLevel
		throw Exception("", -1, ErrorLevel) ;note: Exception.Extra is set to the number of failures
}
Float(Value)
{
	if Value is % "number"
		return Value + 0.0
	throw Exception("Type mismatch.", -1)
}
FormatTime(YYYYMMDDHH24MISS:="", Format:="")
{
	local OutputVar
	if StrLen(YYYYMMDDHH24MISS)
	{
		if YYYYMMDDHH24MISS is not % "time"
			throw Exception("Parameter #1 is invalid.", -1)
		if (StrLen(YYYYMMDDHH24MISS) & 1) || (StrLen(YYYYMMDDHH24MISS) > 14)
			throw Exception("Parameter #1 is invalid.", -1)
	}
	FormatTime, OutputVar, % YYYYMMDDHH24MISS, % Format
	return OutputVar
}
GroupActivate(GroupName, Mode:="")
{
	GroupActivate, % GroupName, % Mode
	if ErrorLevel
		return 0 + 0
	;note: AHK v1's GroupActivate doesn't seem to set the last found window, so we use WinExist():
	return 0 + WinExist("ahk_group " GroupName)
}
GroupAdd(GroupName, WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hWnd
	if hWnd := AHKFC_WinExistSingleAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		GroupAdd, % GroupName, % "ahk_id " hWnd
	else
		GroupAdd, % GroupName, % WinTitle, % WinText, % ExcludeTitle, % ExcludeText
}
GroupClose(GroupName, Mode:="")
{
	GroupClose, % GroupName, % Mode
}
GroupDeactivate(GroupName, Mode:="")
{
	GroupDeactivate, % GroupName, % Mode
}
HotIf(Params*) ;ExpressionOrFunction
{
	if Params.Length()
		Hotkey, % "If", % Params[1]
	else
		Hotkey, % "If"
}
HotIfWinActive(Params*) ;WinTitle, WinText
{
	if Params.Length()
		AHKFC_HotIfWin(A_ThisFunc, Params*)
	else
		Hotkey, % "If"
}
HotIfWinExist(Params*) ;WinTitle, WinText
{
	if Params.Length()
		AHKFC_HotIfWin(A_ThisFunc, Params*)
	else
		Hotkey, % "If"
}
HotIfWinNotActive(Params*) ;WinTitle, WinText
{
	if Params.Length()
		AHKFC_HotIfWin(A_ThisFunc, Params*)
	else
		Hotkey, % "If"
}
HotIfWinNotExist(Params*) ;WinTitle, WinText
{
	if Params.Length()
		AHKFC_HotIfWin(A_ThisFunc, Params*)
	else
		Hotkey, % "If"
}
Hotkey(KeyName, Action:="", Options:="")
{
	;if !InStr(Options, "UseErrorLevel")
	;	Options .= " UseErrorLevel"
	if InStr(Options, "UseErrorLevel")
		Options := RegExReplace(Options, "i)UseErrorLevel") ;RegExReplace preferable to StrReplace, since unaffected by A_StringCaseSense
	try Hotkey, % KeyName, % Action, % Options
	catch
		throw Exception("", -1)
	if ErrorLevel
		throw Exception("", -1)
	;if (InStr(KeyName, "IfWin") || InStr(Options, "UseErrorLevel"))
	;	return !ErrorLevel
}
ImageSearch(ByRef OutputVarX:="", ByRef OutputVarY:="", X1:="", Y1:="", X2:="", Y2:="", ImageFile:="")
{
	if !InStr(ImageFile, "HBITMAP:")
	&& !InStr(ImageFile, "HICON:")
	&& !FileExist(RegExReplace(ImageFile, "^.*[ `t](?=.*:)")) ;file not found
		throw Exception("Parameter #7 is invalid.", -1)
	ImageSearch, OutputVarX, OutputVarY, % X1, % Y1, % X2, % Y2, % ImageFile
	if (ErrorLevel == 2)
		throw Exception("", -1)
	return !ErrorLevel
}
IniDelete(Params*) ;Filename, Section, Key
{
	local Filename, Key, Section
	if (Params.Length() > 3)
		throw Exception("Too many parameters passed to function.", -1)
	Filename := Params[1]
	Section := Params[2]
	Key := Params[3]
	if (Params.Length() == 3)
		IniDelete, % Filename, % Section, % Key
	else if (Params.Length() == 2)
		IniDelete, % Filename, % Section
	if ErrorLevel
		throw Exception("", -1)
}
IniRead(Filename, Section:="`f`a`b", Key:="`f`a`b", Default:="`f`a`b")
{
	local OutputVar
	if (Section != "`f`a`b") && (Key != "`f`a`b")
	{
		;note: if Default is set to '', then 'ERROR' is returned on failure
		;note: if Default is set to ' ', then '' is returned on failure
		;note: the AHK v2 IniRead function only returns a trimmed default value, if Key is specified and it fails to find Key, (it returns the default value unchanged, if it fails to retrieve a section/list of section names)
		if (Default == "")
			Default := " "
		IniRead, OutputVar, % Filename, % Section, % Key, % Default
	}
	else if (Section == "`f`a`b")
	{
		if (Key != "`f`a`b")
			throw Exception("Parameter #2 of IniRead must not be omitted in this case.", -1)
		else if !FileExist(Filename)
			OutputVar := Default
		else
			IniRead, OutputVar, % Filename ;note: blank on error, should only fail (in theory) if the file doesn't exist
	}
	else if (Key == "`f`a`b")
	{
		IniRead, OutputVar, % Filename, % Section ;note: blank on error
		if (OutputVar == "") ;blank, so try to determine if section is blank or non-existent:
		{
			FileRead, OutputVar, % Filename
			if InStr(OutputVar, "`n[" Section "]")
			|| (InStr(OutputVar, "[" Section "]") == 1)
				OutputVar := ""
			else
				OutputVar := Default
		}
	}

	;note: the Winapi function (GetPrivateProfileString) trims trailing spaces (but not tabs) from the output (i.e. it reads the key value or uses the default, and trims it)
	;note: ErrorLevel is not set by the AHK v1 IniRead command
	if (OutputVar == "`f`a`b")
		throw Exception("The requested key, section or file was not found.", -1)
	return OutputVar
}
IniWrite(Value, Filename, Section, Params*) ;Value, Filename, Section, Key
{
	if (Params.Length() > 1)
		throw Exception("Too many parameters passed to function.", -1)
	if Params.HasKey(1)
	{
		if RegExMatch(Params[1], "^[[;]") || InStr(Params[1], "=")
			MsgBox, % "IniWrite:`r`n" "custom warning`r`n:" "ini key '" Params[1] "' contains one of: [ `; ="
		IniWrite, % Value, % Filename, % Section, % Params[1]
	}
	else
		IniWrite, % Value, % Filename, % Section ;Pairs
	if ErrorLevel
		throw Exception("", -1)
}
;[FIXME] remove AHKFC_InputBoxSimple/AHKFC_InputBoxObj when appropriate
InputBox(Prompt:="", Title:="`f`a`b", Options:="", Default:="", ByRef Result:="")
{
	;return AHKFC_InputBoxSimple(Prompt, Title, Options, Default, Result) ;returns Value (Result as ByRef parameter)
	return AHKFC_InputBoxObj(Prompt, Title, Options, Default) ;returns Value/Result as object
}
AHKFC_InputBoxSimple(Prompt:="", Title:="`f`a`b", Options:="", Default:="", ByRef Result:="") ;Prompt, Title, Options, Default, Result ;4 standard params + custom Result param
{
	local _, _Err, _H := "", _P, _T, _W := "", _X := "", _Y := "", OutputVar
	;local Default, Options, Prompt, Title

	;if (Params.Length() > 4)
	;	throw Exception("Too many parameters passed to function.", -1)
	;Prompt := Params[1]
	;Title := !Params.HasKey(2) ? A_ScriptName : ((Params[2] == "") ? " " : Params[2])
	;Options := Params[3]
	;Default := Params[4]

	;note: "`f`a`b": arbitrary default value (equivalent to 'unset' in AHK v2):
	Title := (Title = "`f`a`b") ? A_ScriptName : ((Title == "") ? " " : Title)

	;AHK v2 validates the value of a particular option:
	;X and Y = integer (can be negative)
	;W and H = positive integer only
	;T = positive integer/float
	;Credits to Lexikos [https://goo.gl/VjMTYu , https://goo.gl/ebEjon]
	RegExMatch(Options, "i)^[ \t]*(?:(?:X(?<X>-?\d+)|Y(?<Y>-?\d+)|W(?<W>\d+)"
	. "|H(?<H>\d+)|T(?<T>\d+(?:\.\d+)?)|(?<P>Password\S?)"
	. "|(?<Err>\S+)(*ACCEPT)"
	. ")(?=[ \t]|$)[ \t]*)*$", _)

	if (_Err != "")
		throw Exception("Invalid option.", -1, _Err) ;note: in AHK v2, Exception.Extra is set to the first invalid option and subsequent text (undocumented)

	;custom: resize InputBox if width and height omitted:
	static FuncLargeFont := Func("JEE_InputBoxLargeFont")
	if IsObject(FuncLargeFont) && !StrLen(_W) && !StrLen(_H)
		%FuncLargeFont%(Prompt, _X, _Y, _W, _H)

	InputBox, OutputVar, % Title, % Prompt, % _P ? "HIDE" : "", % _W, % _H, % _X, % _Y,, % _T, % Default
	;InputBox, OutputVar, % Title, % Prompt, % _P ? "HIDE" : "", % _W, % _H, % _X, % _Y, Locale, % _T, % Default ;AHK v1.1.31+
	;note: AHK v2 returns an object, the function here returns a string (use custom function 'AHKFC_InputBoxObj' if you want an object)
	Result := ErrorLevel
	if (Result == 0)
		Result := "OK"
	else if (Result == 1)
		Result := "Cancel"
	else if (Result == 2)
		Result := "Timeout"
	ErrorLevel := Result ;for 'AHKFC_InputBoxObj', note: AHK v2 does not use ErrorLevel
	return OutputVar
}
AHKFC_InputBoxObj(Params*) ;Prompt, Title, Options, Default
{
	if (Params.Length() > 4)
		throw Exception("Too many parameters passed to function.", -1)
	return {Value:AHKFC_InputBoxSimple(Params*), Result:ErrorLevel} ;note: order must be Value then Result
}
Integer(Value)
{
	local Base, Buf, Err, Num, Pfx, Sign
	static FuncStrToI64 := A_IsUnicode ? "msvcrt\_wcstoi64" : "msvcrt\_strtoi64"
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	if Value is % "float"
	{
		VarSetCapacity(Buf, 8, 0)
		return NumGet(NumPut(Value+0.0, &Buf, 0, "UInt64"), -8, "Int64") ;note: rounds towards 0 ;note: '+0.0' needed, e.g. it affects QNAN and INF interpretation
	}

	Sign := 1
	Pfx := ""
	if (SubStr(Value, 1, 1) = "-")
	{
		Sign := -1
		Pfx := "-"
		Value := SubStr(Value, 2)
	}

	;note: unlike AHK v2, this function supports the '0b' prefix ;[FIXME] AHK v2 probably ought to support '0b' (wish list idea)
	;warning: with base '0', _wcstoi64/_strtoi64 treat integers that begin with '0', but not '0x'/'0X', as octal (base 8), thus use bases 10/16 explicitly:
	Base := 10
	if (SubStr(Value, 1, 2) = "0b")
	{
		Base := 2
		Value := SubStr(Value, 3)
		if !RegExMatch(Value, "^[01]+$")
			throw Exception("Type mismatch.", -1)
	}
	else if Value is not % "integer"
		throw Exception("Type mismatch.", -1)
	if (SubStr(Value, 1, 2) = "0x")
	{
		Base := 16
		Value := SubStr(Value, 3)
	}

	DllCall("msvcrt\_set_errno", "Int",0, "Cdecl")
	Num := DllCall(FuncStrToI64, "Str",Pfx Value, "Str","", "Int",Base, "Cdecl Int64")
	Err := 0
	DllCall("msvcrt\_get_errno", "Int*",IsV1?Err:&Err, "Cdecl")
	;e.g. a common error: ERANGE := 34 ;source: errno.h
	if !Err ;if no error, Value is valid Int64:
		return 0 + Num

	;if reach here, Value is a valid integer, but outside the Int64 range, so wraparound/overflow/'truncate':
	;note: the operators apply a wraparound when required:
	Num := 0
	Loop Parse, Value
	{
		Num *= Base
		Num += Sign * Format("{}", "0x" A_LoopField)
	}
	return 0 + Num
}
IsAlnum(Value, Mode:="")
{
	local Result, SCS
	SCS := A_StringCaseSense
	StringCaseSense, % (Mode = "Locale") ? "Locale" : "Off"
	Result := False
	if Value is % "alnum"
		Result := True
	StringCaseSense, % SCS
	return 0 + Result
}
IsAlpha(Value, Mode:="")
{
	local Result, SCS
	SCS := A_StringCaseSense
	StringCaseSense, % (Mode = "Locale") ? "Locale" : "Off"
	Result := False
	if Value is % "alpha"
		Result := True
	StringCaseSense, % SCS
	return 0 + Result
}
IsDigit(Value)
{
	if Value is % "digit"
		return 0 + True
	return 0 + False
}
IsFloat(Value)
{
	if Value is % "float"
		return 0 + True
	return 0 + False
}
IsInteger(Value)
{
	if Value is % "integer"
		return 0 + True
	return 0 + False
}
IsLower(Value, Mode:="")
{
	local Result, SCS
	SCS := A_StringCaseSense
	StringCaseSense, % (Mode = "Locale") ? "Locale" : "Off"
	Result := False
	if Value is % "lower"
		Result := True
	StringCaseSense, % SCS
	return 0 + Result
}
IsNumber(Value)
{
	if Value is % "number"
		return 0 + True
	return 0 + False
}
IsSpace(Value)
{
	if Value is % "space"
		return 0 + True
	return 0 + False
}
IsTime(Value)
{
	if (StrLen(Value) & 1) || (StrLen(Value) > 14) ;note: 'is' already fails for < 4 chars (a year below 1601 fails), valid lengths: 4/6/8/10/12/14
		return 0 + False
	if Value is % "time"
		return 0 + True
	return 0 + False
}
IsUpper(Value, Mode:="")
{
	local Result, SCS
	SCS := A_StringCaseSense
	StringCaseSense, % (Mode = "Locale") ? "Locale" : "Off"
	Result := False
	if Value is % "upper"
		Result := True
	StringCaseSense, % SCS
	return 0 + Result
}
IsXDigit(Value)
{
	if Value is % "xdigit"
		return 0 + True
	return 0 + False
}
KeyHistory(MaxEvents:="")
{
	if (MaxEvents != "") ;[FIXME] backport function to AHK v1 (wish list idea)
		throw Exception("not handled by library", -1)
	KeyHistory
}
KeyWait(KeyName, Options:="")
{
	KeyWait, % KeyName, % Options
	return !ErrorLevel
}
ListHotkeys()
{
	ListHotkeys
}
ListLines(Mode:="")
{
	local OldValue := A_ListLines
	if (Mode == "")
		ListLines
	else if Mode in % "1,0"
		ListLines, % Mode ? "On" : "Off"
	else
		throw Exception("Parameter #1 is invalid.", -1) ;[FIXME] AHK v2 doesn't throw for invalid mode (wish list idea)
	return OldValue
}
ListVars()
{
	;limitation: won't work if called from within a function
	global
	ListVars
}
ListViewGetContent(Options:="", Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl, OutputVar
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlGet, OutputVar, % "List", % Options,, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	return OutputVar
}
;Map (see lower down)
MenuSelect(WinTitle:="", WinText:="", Menu:="", SubMenu1:="", SubMenu2:="", SubMenu3:="", SubMenu4:="", SubMenu5:="", SubMenu6:="", ExcludeTitle:="", ExcludeText:="")
{
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinMenuSelectItem,,, % Menu, % SubMenu1, % SubMenu2, % SubMenu3, % SubMenu4, % SubMenu5, % SubMenu6
	if ErrorLevel
		throw Exception("", -1)
}
MonitorGet(N:="", ByRef OutLeft:="", ByRef OutTop:="", ByRef OutRight:="", ByRef OutBottom:="") ;note: 'Out' prefix is necessary for use with SysGet
{
	;note: SysGet creates 4 variables: e.g. Out -> OutLeft/OutTop/OutRight/OutBottom
	local Out
	SysGet, Out, % "Monitor", % N
	if (N == "")
		SysGet, N, % "MonitorPrimary"
	return 0 + N
}
MonitorGetCount()
{
	local OutputVar
	SysGet, OutputVar, % "MonitorCount"
	return 0 + OutputVar
}
MonitorGetName(N:="")
{
	local OutputVar
	SysGet, OutputVar, % "MonitorName", % N
	return OutputVar
}
MonitorGetPrimary()
{
	local OutputVar
	SysGet, OutputVar, % "MonitorPrimary"
	return 0 + OutputVar
}
MonitorGetWorkArea(N:="", ByRef OutLeft:="", ByRef OutTop:="", ByRef OutRight:="", ByRef OutBottom:="") ;note: 'Out' prefix is necessary for use with SysGet
{
	;note: SysGet creates 4 variables: e.g. Out -> OutLeft/OutTop/OutRight/OutBottom
	local Out
	SysGet, Out, % "MonitorWorkArea", % N
	if (N == "")
		SysGet, N, % "MonitorPrimary"
	return 0 + N
}
MouseClick(WhichButton:="Left", X:="", Y:="", ClickCount:="", Speed:="", DownOrUp:="", Relative:="")
{
	MouseClick, % WhichButton, % X, % Y, % ClickCount, % Speed, % DownOrUp, % Relative
}
MouseClickDrag(WhichButton, X1:="", Y1:="", X2:="", Y2:="", Speed:="", Relative:="")
{
	MouseClickDrag, % WhichButton, % X1, % Y1, % X2, % Y2, % Speed, % Relative
}
MouseGetPos(ByRef OutputVarX:="", ByRef OutputVarY:="", ByRef OutputVarWin:="", ByRef OutputVarControl:="", Flag:=0)
{
	MouseGetPos, OutputVarX, OutputVarY, OutputVarWin, OutputVarControl, % Flag
	OutputVarWin += 0
	if (Flag & 2)
		OutputVarControl += 0
	;note: no return value, returns values via the ByRef variables
}
MouseMove(X, Y, Speed:="", Relative:="")
{
	MouseMove, % X, % Y, % Speed, % Relative
}
MsgBox(Params*) ;Text, Title, Options
{
	local Match, Options, Result, Temp, Text, Timeout, Title, Type
	static IsReady, TypeArray

	;custom: large MsgBox if function available:
	;static FuncLargeFont := Func("JEE_MsgBoxLargeFont")
	static FuncLargeFont := Func("JEE_MsgBoxPure")
	if IsObject(FuncLargeFont)
		return %FuncLargeFont%(Params*)

	if (Params.Length() > 3)
		throw Exception("Too many parameters passed to function.", -1)
	if !VarSetCapacity(IsReady)
	{
		TypeArray := {"OK":0, "O":0, "OKCancel":1, "O/C":1, "OC":1, "AbortRetryIgnore":2, "A/R/I":2, "ARI":2
		, "YesNoCancel":3, "Y/N/C":3, "YNC":3, "YesNo":4, "Y/N":4, "YN":4, "RetryCancel":5, "R/C":5, "RC":5
		, "CancelTryAgainContinue":6, "C/T/C":6, "CTC":6, "Iconx":16, "Icon?":32, "Icon!":48, "Iconi":64
		, "Default2":256, "Default3":512, "Default4":768}
		IsReady := "1"
	}

	Title := !Params.HasKey(2) ? A_ScriptName : (Params[2] == "") ? " " : Params[2]
	Options := Params[3]
	Timeout := ""
	Type := 0
	if Options
	{
		Loop Parse, Options, % " `t"
			(Temp := Abs(A_LoopField)) || (Temp := TypeArray[A_LoopField]) ? (Type |= Temp)
			: RegExMatch(A_LoopField, "Oi)^T(\d+\.?\d*)$", Match) ? Timeout := Match[1]
			: 0
	}

	;Text := !Params.Length() ? "Press OK to continue." : Params.HasKey(1) ? Params[1] : ""
	Text := Params.HasKey(1) ? Params[1] : (Type & 0xF) ? "" : "Press OK to continue." ;Use default text only if OK is the only button.
	Text := StrReplace(Text, "`r`n", "`n") ;custom: text replacement to avoid incorrect CRs/LFs when copy the text

	MsgBox, % Type, % Title, % Text, % Timeout
	Loop Parse, % "Timeout,OK,Cancel,Yes,No,Abort,Ignore,Retry,Continue,TryAgain", % ","
	{
		IfMsgBox, % Result := A_LoopField
			break
	}
	return Result
}
;MsgBox(Text)
;{
;	MsgBox, % Text
;}
Number(Value)
{
	if !IsNumber(Value)
		throw Exception("Parameter #1 of Number.Call requires a Number, but received a non-number.`n`n" "Specifically: " Value, -1)
	else if IsFloat(Value)
		return Float(Value)
	else
		return Integer(Value)
}
ObjFromPtr(Ptr)
{
	local Obj
	if IsObject(Ptr)
		throw Exception("Parameter #1 is invalid.", -1)
	Obj := Object(Ptr) ;'Object(Ptr)' increments the reference count, and returns an object
	ObjRelease(Ptr)
	return Obj
}
ObjFromPtrAddRef(Ptr)
{
	if IsObject(Ptr)
		throw Exception("Parameter #1 is invalid.", -1)
	return Object(Ptr) ;'Object(Ptr)' increments the reference count, and returns an object
}
ObjPtr(Obj)
{
	if !IsObject(Obj)
		throw Exception("Expected an object.", -1)
	return &Obj
}
ObjPtrAddRef(Obj)
{
	if !IsObject(Obj)
		throw Exception("Expected an object.", -1)
	return Object(Obj) ;'Object(Obj)' increments the reference count, and returns an address
	;return ObjAddRef(&Obj)*0 + &Obj ;equivalent to line above
}
OutputDebug(Text)
{
	OutputDebug, % Text
}
Pause(NewState:="")
{
	;former code:
	;if NewState in % "1,0,-1" ;On,Off,Toggle
	;	NewState := NewState == -1 ? "Toggle" : NewState ? "On" : "Off"
	;Pause, % NewState, % OperateOnUnderlyingThread

	if NewState in % "On,Off,Toggle"
		throw Exception("Parameter #1 is invalid.", -1)
	if (NewState == "")
		Pause, % "On"
	else if (NewState == 1)
		Pause, % "On", 1
	else if (NewState == 0)
		Pause, % "Off"
	else if (NewState == -1)
		Pause, % "Toggle", 1
}
PixelGetColor(X, Y, Mode:="")
{
	local OutputVar
	PixelGetColor, OutputVar, % X, % Y, % Mode " RGB" ;v2 uses RGB
	if ErrorLevel
		throw Exception("", -1)
	;[FIXME] AHK v2 returns a string, an integer might be more logical (wish list idea)
	return OutputVar
}
PixelSearch(ByRef OutputVarX:="", ByRef OutputVarY:="", X1:="", Y1:="", X2:="", Y2:="", ColorID:="", Variation:=0)
{
	PixelSearch, OutputVarX, OutputVarY, % X1, % Y1, % X2, % Y2, % ColorID, % Variation, % "Fast RGB"
	if (ErrorLevel == 2)
		throw Exception("", -1)
	return !ErrorLevel
}
;PostMessage(Msg, wParam, lParam, Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="") ;custom variant: requires that the first 3 parameters be specified
PostMessage(Msg, wParam:="", lParam:="", Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl
	if IsObject(wParam)
	{
		wParam := wParam.Ptr
		if (wParam == "")
			throw Exception("", -1)
	}
	else if !StrLen(wParam)
		wParam := 0
	if IsObject(lParam)
	{
		lParam := lParam.Ptr
		if (lParam == "")
			throw Exception("", -1)
	}
	else if !StrLen(lParam)
		lParam := 0
	if wParam is not % "number"
		throw Exception("", -1)
	if lParam is not % "number"
		throw Exception("", -1)

	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	PostMessage, % Msg, % wParam, % lParam,, % "ahk_id " hCtl
	if (ErrorLevel = "FAIL")
		throw Exception("", -1)
	;note: no return value
}
;Pow(Base, Exponent)
;{
;	return Base ** Exponent
;}
ProcessClose(PIDOrName)
{
	Process, % "Close", % PIDOrName
	;note: AHK v2: does not throw
	;note: AHK v2: returns PID or 0
	return 0 + ErrorLevel
}
ProcessExist(PIDOrName:="")
{
	Process, % "Exist", % PIDOrName
	;note: AHK v2: returns PID or 0
	return 0 + ErrorLevel
}
ProcessGetName(PIDOrName:="")
{
	local DHW, hWnd, oProc, PID, PName
	if (PIDOrName == "")
	{
		SplitPath, A_AhkPath, PName
		return PName
	}

	Process, % "Exist", % PIDOrName
	PID := ErrorLevel
	if !PID
		throw Exception("", -1)

	DHW := A_DetectHiddenWindows
	DetectHiddenWindows, % "On"
	hWnd := WinExist("ahk_pid " PID)
	if hWnd
	{
		WinGet, PName, % "ProcessName", % "ahk_id " hWnd
		DetectHiddenWindows, % DHW
		return PName
	}
	DetectHiddenWindows, % DHW

	for oProc in ComObjGet("winmgmts:").ExecQuery("Select * from Win32_Process where ProcessId=" PID)
		return oProc.Name
	throw Exception("", -1)
}
ProcessGetPath(PIDOrName:="")
{
	local DHW, hWnd, oProc, PID, PPath
	if (PIDOrName == "")
		return A_AhkPath

	Process, % "Exist", % PIDOrName
	PID := ErrorLevel
	if !PID
		throw Exception("", -1)

	DHW := A_DetectHiddenWindows
	DetectHiddenWindows, % "On"
	hWnd := WinExist("ahk_pid " PID)
	if hWnd
	{
		WinGet, PPath, % "ProcessPath", % "ahk_id " hWnd
		DetectHiddenWindows, % DHW
		return PPath
	}
	DetectHiddenWindows, % DHW

	for oProc in ComObjGet("winmgmts:").ExecQuery("Select * from Win32_Process where ProcessId=" PID)
		return oProc.ExecutablePath
	throw Exception("", -1)
}
ProcessGetParent(PIDOrName:="")
{
	local oProc, PID
	Process, % "Exist", % PIDOrName
	PID := ErrorLevel
	if !PID
		throw Exception("", -1)

	for oProc in ComObjGet("winmgmts:").ExecQuery("Select * from Win32_Process where ProcessId=" PID)
		return oProc.ParentProcessId
	throw Exception("", -1)
}
ProcessSetPriority(Level, PIDOrName:="")
{
	Process, % "Priority", % PIDOrName, % Level
	;note: AHK v2: returns PID or 0
	return 0 + ErrorLevel
}
ProcessWait(PIDOrName, Timeout:="")
{
	Process, % "Wait", % PIDOrName, % Timeout
	;note: AHK v2: returns PID or 0
	return 0 + ErrorLevel
}
ProcessWaitClose(PIDOrName, Timeout:="")
{
	Process, % "WaitClose", % PIDOrName, % Timeout
	;note: AHK v2: returns PID or 0
	;note: warning: counterintuitive: returns 0 (if all processes closed), or PID (of first-found still-extant process)
	return 0 + ErrorLevel
}
Random(vNum1:="`f`a`b", vNum2:="`f`a`b")
{
	;note: unlike AHK v2, this function always uses seeding ;[FIXME] AHK v2 probably ought to restore seeding (wish list idea)
	;WARNING: keep '-0x8000000000000000' intact, e.g. '- 0x8000000000000000' will silently give an incorrect value in AHK v1
	local vDiff, vErrorMargin, vIsMissing1, vIsMissing2, vRand
	;local oBuf

	;oBuf := Buffer(8)
	;DllCall("advapi32\SystemFunction036", "Ptr",oBuf.Ptr, "UInt",oBuf.Size) ;RtlGenRandom
	;vRand := NumGet(oBuf.Ptr, 0, "UInt64")

	vIsMissing1 := vIsMissing2 := 0
	if (vNum1 = "`f`a`b")
	{
		vIsMissing1 := 1
		vNum1 := 0
	}
	if (vNum2 = "`f`a`b")
	{
		vIsMissing2 := 1
		vNum2 := 0
	}
	if !IsNumber(vNum1)
		throw Exception("", -1)
	if !IsNumber(vNum2)
		throw Exception("", -1)

	if IsFloat(vNum1) || IsFloat(vNum2) || (vIsMissing1 && vIsMissing2)
	{
		if (vIsMissing1 && vIsMissing2)
			vNum2 := 1.0
		vNum1 += 0.0
		vNum2 += 0.0

		if (vNum1 > vNum2)
			BIFEx_Swap(vNum1, vNum2)

		vRand := BIFEx_BitShiftRightLogical(AHKFC_RandomInt64(), 11)
		;note: 0xFFFFFFFFFFFFFFFF >>> 11 = 0x1FFFFFFFFFFFFF
		;note: 9007199254740992 = 0x20000000000000
		;so max vRand / 9007199254740992 = 0x1FFFFFFFFFFFFF / 0x20000000000000
		return (vRand/9007199254740992.0)*(vNum2-vNum1) + vNum1
	}
	else
	{
		if (vNum1 > vNum2)
			BIFEx_Swap(vNum1, vNum2)

		vDiff := vNum2 - vNum1
		if (vDiff <= 0xFFFFFFFF)
		&& (vDiff > 0)
		{
			if (vNum1 >= -0x80000000) && (vNum2 <= 0x7FFFFFFF)
			{
				Random, vRand, % vNum1, % vNum2
				return 0 + vRand
			}
			else if (vNum1 >= 0) && (vNum2 <= 0xFFFFFFFF)
			{
				Random, vRand, % vNum1 - 0x80000000, % vNum2 - 0x80000000
				return vRand + 0x80000000
			}
			else
			{
				Random, vRand, -0x80000000, % -0x80000000 + vDiff
				return vRand + 0x80000000 + vNum1
			}
		}
		else if (vNum1 == -0x8000000000000000) && (vNum2 == 0x7FFFFFFFFFFFFFFF)
			return AHKFC_RandomInt64()
		else
		{
			vRand := AHKFC_RandomInt64()
			;What we actually want is (UINT64_MAX + 1) % (u_max + 1), but without overflow.
			vErrorMargin := BIFEx_ModUInt64(-1, vDiff+1)
			if (vErrorMargin != vDiff) ;i.e. ((vErrorMargin + 1) % (u_max + 1)) != 0.
			{
				++vErrorMargin
				;error_margin is now the remainder after dividing (UINT64_MAX+1) by (u_max+1).
				;This is also the size of the incomplete number set which must be excluded to
				;ensure even distribution of random numbers.  For simplicity we just take the
				;number set starting at 0, since use of modulo below will auto-correct.
				;For example, with a target range of 1..100, the error_margin should be 16,
				;which gives a mere 16 in 2**64 chance that a second iteration will be needed.
				while (vRand < vErrorMargin)
				{
					;To ensure even distribution, keep trying until outside the error margin.
					vRand := AHKFC_RandomInt64()
				}
			}
			vRand := BIFEx_ModUInt64(vRand, vDiff+1)
			return vRand + vNum1
		}
	}
}
;RandomSimple(A:="`f`a`b", B:="`f`a`b")
;{
;	local OutputVar
;	if (A = "`f`a`b" && B = "`f`a`b")
;	{
;		Random, OutputVar, -0x80000000, 0x7FFFFFFF
;		return (OutputVar+0x80000000) / 0x100000000 ;a float: 'int between 0 and 0xFFFFFFFF' / 0x100000000
;	}
;	if (A = "`f`a`b")
;		A := 0
;	if (B = "`f`a`b")
;		B := 0
;	if (B >= A)
;		Random, OutputVar, % A, % B
;	else
;		Random, OutputVar, % B, % A
;	return 0 + OutputVar
;}
RandomSeed(NewSeed)
{
	Random,, % NewSeed ;note: initial comma appears to be necessary
}
RegCreateKey(KeyName:="`f`a`b")
{
	local Flags, hKey, hRootKey, RootKey, SubKey
	static oRootKey := Object("HKEY_CLASSES_ROOT",0x80000000, "HKEY_CURRENT_USER",0x80000001, "HKEY_LOCAL_MACHINE",0x80000002, "HKEY_USERS",0x80000003, "HKEY_CURRENT_CONFIG",0x80000005, "HKEY_DYN_DATA",0x80000006, "HKCR",0x80000000, "HKCU",0x80000001, "HKLM",0x80000002, "HKU",0x80000003, "HKCC",0x80000005)
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)

	if (KeyName = "`f`a`b")
	{
		if !StrLen(A_LoopRegKey)
			throw Exception("", -1)
		KeyName := A_LoopRegKey
		if StrLen(A_LoopRegSubKey)
			KeyName .= "\" A_LoopRegSubKey
		if (A_LoopRegType = "KEY") && StrLen(A_LoopRegName)
			KeyName .= "\" A_LoopRegName
	}

	if !InStr(KeyName, "\")
		throw Exception("", -1)

	RootKey := RegExReplace(KeyName, "\\.*")
	SubKey := RegExReplace(KeyName, "^.*?(\\|$)")
	if !oRootKey.HasKey(RootKey)
		throw Exception("", -1)
	hRootKey := oRootKey[RootKey]

	Flags := 0x20007 ;KEY_WRITE := 0x20006 ;KEY_QUERY_VALUE := 0x1
	if (A_RegView == 64)
		Flags |= 0x100 ;KEY_WOW64_64KEY := 0x100
	else if (A_RegView == 32)
		Flags |= 0x200 ;KEY_WOW64_32KEY := 0x200

	hKey := 0
	;REG_OPTION_NON_VOLATILE := 0x0
	;source code specifies blank string for lpClass, DllCall below specifies null:
	if DllCall("advapi32\RegCreateKeyEx" (A_IsUnicode?"W":"A"), "Ptr",hRootKey, "Str",SubKey, "UInt",0, "Ptr",0, "UInt",0x0, "UInt",Flags, "Ptr",0, "Ptr*",IsV1?hKey:&hKey, "Ptr",0, "UInt")
		throw Exception("", -1)
	DllCall("advapi32\RegCloseKey", "Ptr",hKey)
}
RegDelete(Params*) ;KeyName, ValueName
{
	;note: StrLen(A_LoopRegKey) indicates whether inside a reg loop or not
	if (Params.Length() > 2)
		throw Exception("Too many parameters passed to function.", -1)
	if !Params.HasKey(2) && StrLen(A_LoopRegKey)
		Params[2] := A_LoopRegName
	if (Params[2] == "")
		Params[2] := "AHK_DEFAULT"
	if Params.HasKey(1)
	{
		if InStr(Params[1], "\")
			RegDelete, % Params[1], % Params[2]
		else
			RegDelete, % Params[1],, % Params[2]
	}
	else if (A_LoopRegType = "KEY")
		RegDelete, % A_LoopRegKey, % A_LoopRegSubKey "" (StrLen(A_LoopRegSubKey) ? "\" : "") A_LoopRegName, % Params[2]
	else
		RegDelete, % A_LoopRegKey, % A_LoopRegSubKey, % Params[2]
	if ErrorLevel
		throw Exception("", -1)
}
RegDeleteKey(Params*) ;KeyName
{
	;note: StrLen(A_LoopRegKey) indicates whether inside a reg loop or not
	if (Params.Length() > 1)
		throw Exception("Too many parameters passed to function.", -1)
	if Params.HasKey(1)
		RegDelete, % Params[1]
	else if (A_LoopRegKey == "")
		throw Exception("", -1)
	else if (A_LoopRegType = "KEY")
		RegDelete
	else if StrLen(A_LoopRegSubKey)
		RegDelete, % A_LoopRegKey, % A_LoopRegSubKey
	;else if StrLen(A_LoopRegKey)
	;	RegDelete, % A_LoopRegKey ;[FIXME] would this delete one of HKLM/HKU/HKCU/HKCR/HKCC?
	else
		ErrorLevel := 1
	if ErrorLevel
		throw Exception("", -1)
}
RegRead(KeyName:="`f`a`b", ValueName:="`f`a`b", Default:="`f`a`b")
{
	local OutputVar
	;note: StrLen(A_LoopRegKey) indicates whether inside a reg loop or not
	if (ValueName = "`f`a`b")
	{
		if StrLen(A_LoopRegKey) && (A_LoopRegType != "KEY")
			ValueName := A_LoopRegName
		else
			ValueName := ""
	}
	if (KeyName != "`f`a`b")
	{
		if InStr(KeyName, "\")
			RegRead, OutputVar, % KeyName, % ValueName
		else
			RegRead, OutputVar, % KeyName,, % ValueName
	}
	else if (A_LoopRegType = "KEY")
		RegRead, OutputVar, % A_LoopRegKey, % A_LoopRegSubKey "" (StrLen(A_LoopRegSubKey) ? "\" : "") A_LoopRegName, % ValueName
	else if StrLen(A_LoopRegKey)
		RegRead, OutputVar, % A_LoopRegKey, % A_LoopRegSubKey, % ValueName
	else
		throw Exception("Parameter #1 is invalid.", -1)
	if ErrorLevel
	{
		if (Default != "`f`a`b")
			return Default
		throw Exception("", -1)
	}
	return OutputVar
}
RegWrite(Params*) ;Value, ValueType, KeyName, ValueName
{
	;note: StrLen(A_LoopRegKey) indicates whether inside a reg loop or not
	if (Params.Length() > 4)
		throw Exception("Too many parameters passed to function.", -1)
	if !Params.HasKey(3)
	&& ((A_LoopRegKey == "") || (A_LoopRegType = "KEY"))
		throw Exception("Parameter #3 is invalid.", -1)
	if !Params.HasKey(2)
	{
		if (A_LoopRegKey == "") || (A_LoopRegType = "KEY")
			throw Exception("Parameter #2 is invalid.", -1)
		else
			Params[2] := A_LoopRegType
	}
	if !Params.HasKey(4)
	{
		if (A_LoopRegKey == "") || (A_LoopRegType = "KEY")
			Params[4] := ""
		else
			Params[4] := A_LoopRegName
	}
	if !Params.HasKey(1)
	{
		if InStr(A_LoopRegType, "_SZ")
			Params[1] := ""
		else
			Params[1] := 0
	}
	if Params.HasKey(3)
	{
		if InStr(Params[3], "\")
			RegWrite, % Params[2], % Params[3], % Params[4], % Params[1]
		else
			RegWrite, % Params[2], % Params[3],, % Params[4], % Params[1]
	}
	else
		RegWrite, % Params[2], % A_LoopRegKey, % A_LoopRegSubKey, % Params[4], % Params[1]
	if ErrorLevel
		throw Exception("", -1)
}
Reload()
{
	Reload
}
Run(Target, WorkingDir:="", Options:="", ByRef OutputVarPID:="")
{
	if !InStr(Options, "UseErrorLevel")
		Options .= " UseErrorLevel"
	;if InStr(Options, "UseErrorLevel")
	;	Options := RegExReplace(Options, "i)UseErrorLevel") ;RegExReplace preferable to StrReplace, since unaffected by A_StringCaseSense
	Run, % Target, % WorkingDir, % Options, OutputVarPID
	if ErrorLevel && InStr(Options, "UseErrorLevel")
		throw Exception("", -1)
	;note: no return value, returns values via the ByRef variable
}
RunAs(User:="", Password:="", Domain:="")
{
	RunAs, % User, % Password, % Domain
}
RunWait(Target, WorkingDir:="", Options:="", ByRef OutputVarPID:="")
{
	if !InStr(Options, "UseErrorLevel")
		Options .= " UseErrorLevel"
	RunWait, % Target, % WorkingDir, % Options, OutputVarPID
	if (ErrorLevel = "ERROR")
		throw Exception("", -1)
	return 0 + ErrorLevel
}
Send(Keys)
{
	Send, % Keys
}
SendEvent(Keys)
{
	SendEvent, % Keys
}
SendInput(Keys)
{
	SendInput, % Keys
}
SendLevel(Level)
{
	local OldValue := A_SendLevel
	SendLevel, % Level
	return OldValue
}
;SendMessage(Msg, wParam, lParam, Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="", Timeout:="") ;custom variant: requires that the first 3 parameters be specified
SendMessage(Msg, wParam:="", lParam:="", Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="", Timeout:="")
{
	local hCtl
	if IsObject(wParam)
	{
		wParam := wParam.Ptr
		if (wParam == "")
			throw Exception("", -1)
	}
	else if !StrLen(wParam)
		wParam := 0
	if IsObject(lParam)
	{
		lParam := lParam.Ptr
		if (lParam == "")
			throw Exception("", -1)
	}
	else if !StrLen(lParam)
		lParam := 0
	if wParam is not % "number"
		throw Exception("", -1)
	if lParam is not % "number"
		throw Exception("", -1)

	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	SendMessage, % Msg, % wParam, % lParam,, % "ahk_id " hCtl,,,, % Timeout
	if (ErrorLevel = "FAIL")
		throw Exception("", -1)
	return 0 + ErrorLevel
}
SendMode(Mode)
{
	local OldValue := A_SendMode
	SendMode, % Mode
	return OldValue
}
SendPlay(Keys)
{
	SendPlay, % Keys
}
SendText(Keys)
{
	if (StrSplit(A_AhkVersion, ".")[3] >= 27) ;[v1.1.27+]
		SendInput, % "{Text}" Keys
	else
	{
		Keys := StrReplace(Keys, "`r`n", "`n")
		SendInput, % "{Raw}" Keys
	}
}
SetCapsLockState(State:="")
{
	if State in % "1,0"
		State := State ? "On" : "Off"
	SetCapsLockState, % State
}
SetControlDelay(Delay)
{
	local OldValue := A_ControlDelay
	SetControlDelay, % Delay
	return OldValue
}
SetDefaultMouseSpeed(Speed)
{
	local OldValue := A_DefaultMouseSpeed
	SetDefaultMouseSpeed, % Speed
	return OldValue
}
SetKeyDelay(Delay:="", PressDuration:="", Play:="")
{
	SetKeyDelay, % Delay, % PressDuration, % Play
}
SetMouseDelay(Delay, Play:="")
{
	local OldValue := (Play = "Play") ? A_MouseDelayPlay : A_MouseDelay
	SetMouseDelay, % Delay, % Play
	return OldValue
}
SetNumLockState(State:="")
{
	if State in % "1,0"
		State := State ? "On" : "Off"
	SetNumLockState, % State
}
SetRegView(RegView)
{
	local OldValue := A_RegView
	SetRegView, % RegView
	return OldValue
}
SetScrollLockState(State:="")
{
	if State in % "1,0"
		State := State ? "On" : "Off"
	SetScrollLockState, % State
}
SetStoreCapsLockMode(Mode)
{
	local OldValue := A_StoreCapsLockMode
	;note: AHK v2: On/Off not listed, but don't throw
	;if Mode in % "On,Off"
	;	throw Exception("Parameter #1 is invalid.", -1)
	if Mode in % "1,0"
		Mode := Mode ? "On" : "Off"
	SetStoreCapsLockMode, % Mode
	return OldValue
}
SetTimer(Function:="", Period:="", Priority:=0)
{
	if Period in % "On,Off,Delete"
		throw Exception("Parameter #2 is invalid.", -1)
	if (Period == 0)
		Period := "Delete"
	;note: AHK v1: command does not use ErrorLevel
	SetTimer, % Function, % Period, % Priority
}
SetTitleMatchMode(MatchModeOrSpeed)
{
	local OldValue := (MatchModeOrSpeed ~= "i)^(Fast|Slow)$") ? A_TitleMatchModeSpeed : A_TitleMatchMode
	SetTitleMatchMode, % MatchModeOrSpeed
	return OldValue
}
SetWinDelay(Delay)
{
	local OldValue := A_WinDelay
	SetWinDelay, % Delay
	return OldValue
}
SetWorkingDir(DirName)
{
	SetWorkingDir, % DirName
	if ErrorLevel
		throw Exception("Parameter #1 is invalid.", -1)
}
Shutdown(Flag)
{
	;note: AHK v1: command does not use ErrorLevel
	Shutdown, % Flag
}
Sleep(Delay)
{
	Sleep, % Delay
}
Sort(String, Options:="", Function:="`f`a`b")
{
	;note: CLogical mode uses stable sort (like AHK v2)
	;note: the other modes (unfortunately) use unstable sort (unlike AHK v2)
	local SortLogical
	static FuncSortInProgress := 0
	;global zAHKFC_SortFunc ;note: in AHK v1: the local assignment above, makes other function variables global
	SortLogical := !!InStr(Options, "CLogical")
	Options := RegExReplace(Options, "i)(COn|C1)(?= |`t|$)", "C")
	Options := RegExReplace(Options, "i)(COff|C0)(?= |`t|$)")
	Options := RegExReplace(Options, "i)CLocale", "CL")
	Options := RegExReplace(Options, "i)CLogical")

	;[FIXME] unlike v2, throw for an invalid 'F' option (wish list idea):
	if RegExMatch(Options, "(^| |`t)\K[Ff]") ;note: 'DF' and 'COff' are valid in AHK v2
		throw Exception("Parameter #2 is invalid. Use the Function parameter to specify functions.", -1)

	if IsObject(Function)
	{
		if FuncSortInProgress ;[FIXME] can this 'FuncSortInProgress' code be improved/removed?
			MsgBox, % "warning: Sort with function in progress"
		zAHKFC_SortFunc := Function
		Options .= " F AHKFC_SortAux"
		FuncSortInProgress := 1
	}
	else if (Function == "")
		throw Exception("Parameter #3 is invalid.", -1)
	else if (Function != "`f`a`b") ;a non-blank string
	{
		;uncomment these lines to allow function name strings:
		;if IsFunc(Function)
		;	Options .= " F " Function
		;else
		throw Exception("Parameter #3 is invalid.", -1)
	}
	else if SortLogical ;and Function param omitted
		Options .= " F AHKFC_SortLogical"
	Sort, String, % Options
	FuncSortInProgress := 0
	return String
}
SoundBeep(Frequency:=523, Duration:=150)
{
	if (Duration < 0)
		Duration := 150
	SoundBeep, % Frequency, % Duration
}
SoundGetInterface(IID, Component:="", Device:="")
{
	local OutputVar
	OutputVar := BIFEx_Sound(A_ThisFunc, IID, Component, Device)
	if ErrorLevel
		throw Exception("", -1)
	if (OutputVar == "")
		return ""
	return 0 + OutputVar
}
SoundGetMute(Component:="", Device:="")
{
	local OutputVar
	OutputVar := BIFEx_Sound(A_ThisFunc,, Component, Device)
	if ErrorLevel
		throw Exception("", -1)
	if (OutputVar == "")
		return ""
	return 0 + OutputVar
}
SoundGetName(Component:="", Device:="")
{
	local OutputVar
	OutputVar := BIFEx_Sound(A_ThisFunc,, Component, Device)
	if ErrorLevel
		throw Exception("", -1)
	return OutputVar
}
SoundGetVolume(Component:="", Device:="")
{
	local OutputVar
	OutputVar := BIFEx_Sound(A_ThisFunc,, Component, Device)
	if ErrorLevel
		throw Exception("", -1)
	if (OutputVar == "")
		return ""
	return 0.0 + OutputVar
}
SoundPlay(Filename, Wait:="")
{
	SoundPlay, % Filename, % Wait
	if ErrorLevel
		throw Exception("", -1)
}
SoundSetMute(NewSetting, Component:="", Device:="")
{
	BIFEx_Sound(A_ThisFunc, NewSetting, Component, Device)
	if ErrorLevel
		throw Exception("", -1)
}
SoundSetVolume(NewSetting, Component:="", Device:="")
{
	BIFEx_Sound(A_ThisFunc, NewSetting, Component, Device)
	if ErrorLevel
		throw Exception("", -1)
}
SplitPath(_Path, ByRef OutFileName:="", ByRef OutDir:="", ByRef OutExtension:="", ByRef OutNameNoExt:="", ByRef OutDrive:="")
{
	;using '_Path' not 'Path' to avoid error: 'An environment variable is being accessed; see #NoEnv.'
	SplitPath, % _Path, OutFileName, OutDir, OutExtension, OutNameNoExt, OutDrive
	;note: no return value, returns values via the ByRef variables

	;possible future functionality:
	if IsByRef(OutFileName) || IsByRef(OutDir) || IsByRef(OutExtension) || IsByRef(OutNameNoExt) || IsByRef(OutDrive)
		return ""
	return {Path:_Path, Name:OutFileName, Dir:OutDir, Ext:OutExtension, NameNoExt:OutNameNoExt, Drive:OutDrive}
}
;[FIXME] StatusBarGetText/StatusBarWait: add Control support if ever added to AHK v2
StatusBarGetText(PartNum:=1, WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local OutputVar
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	StatusBarGetText, OutputVar, % PartNum
	if ErrorLevel
		throw Exception("", -1)
	return OutputVar
}
StatusBarWait(BarText:="", Timeout:="", PartNum:=1, WinTitle:="", WinText:="", Interval:=50, ExcludeTitle:="", ExcludeText:="")
{
	if (Timeout == 0)
		Timeout := 0.001
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	StatusBarWait, % BarText, % Timeout, % PartNum,,, % Interval
	if (ErrorLevel == 2)
		throw Exception("", -1)
	return !ErrorLevel
}
StrCompare(ByRef String1, ByRef String2, CaseSense:=0) ;ByRef for performance
{
	local Output, SCS, Temp1, Temp2
	SCS := A_StringCaseSense
	if (CaseSense == 0) || (CaseSense = "Off")
		StringCaseSense, % "Off"
	else if (CaseSense == 1) || (CaseSense = "On")
		StringCaseSense, % "On"
	else if (CaseSense = "Locale")
		StringCaseSense, % "Locale"
	else if (CaseSense = "Logical")
	{
		if A_IsUnicode
			return 0 + DllCall("shlwapi\StrCmpLogicalW", "WStr",String1, "WStr",String2)
		Temp1 := String1
		Temp2 := String2
		return 0 + DllCall("shlwapi\StrCmpLogicalW", "WStr",Temp1, "WStr",Temp2)
	}
	else
		throw Error("Parameter #3 is invalid.", -1)
	Output := ("" String1 > "" String2) ? 1 : ("" String1 < "" String2) ? -1 : 0
	StringCaseSense, % SCS
	return 0 + Output
}
String(Value)
{
	if !IsObject(Value)
		return "" Value
	return Value.ToString()
}
StrLower(ByRef String) ;ByRef for performance
{
	local OutputVar := "" ;assign a value to avoid #Warn error ('This variable has not been assigned a value.')
	StringLower, OutputVar, String
	return OutputVar
}
StrPtr(ByRef Value) ;ByRef by necessity, to get input variable address (the function is not ByRef in AHK v2)
{
	static Index, IsReady, StringArray
	if IsObject(Value)
		throw Exception("Expected a String.", -1)
	if IsByRef(Value)
		return &Value

	;note: if a literal string is passed, it is temporarily stored in an array,
	;once 1000 items are reached, array slots are reused:
	if !VarSetCapacity(IsReady)
	{
		StringArray := []
		Index := 0
		IsReady := "1"
	}
	Index := Mod(Index, 1000) + 1
	if (StringArray.Length() < 1000)
		StringArray.Push("" Value)
	else
		StringArray[Index] := "" Value
	return StringArray.GetAddress(Index)
}
StrTitle(ByRef String) ;ByRef for performance
{
	local OutputVar := "" ;assign a value to avoid #Warn error ('This variable has not been assigned a value.')
	StringLower, OutputVar, String, % "T"
	return OutputVar
}
StrUpper(ByRef String) ;ByRef for performance
{
	local OutputVar := "" ;assign a value to avoid #Warn error ('This variable has not been assigned a value.')
	StringUpper, OutputVar, String
	return OutputVar
}
Suspend(Mode:=-1)
{
	if Mode in % "On,Off,Toggle"
		throw Exception("Parameter #1 is invalid.", -1)
	if Mode in % "1,0,-1" ;On,Off,Toggle
		Mode := Mode == -1 ? "Toggle" : Mode ? "On" : "Off"
	Suspend, % Mode
}
SysGet(Property)
{
	local OutputVar
	SysGet, OutputVar, % Property
	return 0 + OutputVar
}
SysGetIPAddresses()
{
	local IPAddress, List
	List := []
	Loop 4
	{
		IPAddress := A_IPAddress%A_Index%
		if (IPAddress != "0.0.0.0")
			List.Push(IPAddress)
	}
	return List
}
Thread(SubFunction, Value1:="", Value2:="")
{
	Thread, % SubFunction, % Value1, % Value2
}
ToolTip(ByRef Text:="", X:="", Y:="", WhichToolTip:=1) ;ByRef for performance
{
	if (WhichToolTip < 1) || (WhichToolTip > 20)
		throw Exception("Parameter #4 is invalid.", -1)
	ToolTip, % Text, % X, % Y, % WhichToolTip
}
TraySetIcon(FileName:="", IconNumber:="", Freeze:="")
{
	if !InStr(FileName, "HBITMAP:")
	&& !InStr(FileName, "HICON:")
	&& (FileName != "*")
	&& !FileExist(FileName)
		throw Exception("Can't load icon.", -1)
	Menu, % "Tray", % "Icon", % FileName, % IconNumber, % Freeze
}
TrayTip(Params*) ;Text, Title, Options
{
	local Num := 0, Options, Text, Title
	if (Params.Length() > 3)
		throw Exception("Too many parameters passed to function.", -1)
	if !Params.Length()
	{
		TrayTip
		return
	}

	Text := !Params.HasKey(1) ? " " : (Params[1] == "") ? " " : Params[1]
	Title := !Params.HasKey(2) ? "" : Params[2]
	Options := Params.HasKey(3) ? Params[3] : 0
	Loop Parse, Options, % " `t"
	{
		(A_LoopField = "Iconi") && (Num |= 1)
		(A_LoopField = "Icon!") && (Num |= 2)
		(A_LoopField = "Iconx") && (Num |= 3)
		(A_LoopField = "Mute") && (Num |= 16)
		if A_LoopField is % "integer"
			Num |= A_LoopField
	}
	TrayTip, % Title, % Text,, % Num
}
Type(Value)
{
	local e, f, m
	static IsReady, nBoundFunc, nEnumObj, nFileObj, nMatchObj
	;note: doesn't fully match the AHK v2 function (sometimes misidentifies a Float as a String, a possible solution: float += 0)
	if !VarSetCapacity(IsReady)
	{
		nMatchObj := NumGet(&(m, RegExMatch("", "O)", m)), 0, "Ptr")
		nBoundFunc := NumGet(&(f := Func("Func").Bind()), 0, "Ptr")
		;nFileObj := NumGet(&(f := FileOpen("*", "w")), 0, "Ptr") ;[FIXME] throwing when in a try block
		nFileObj := NumGet(&(f := FileOpen(A_AhkPath, "r")), 0, "Ptr")
		f := ""
		nEnumObj := NumGet(&(e := ObjNewEnum({})), 0, "Ptr")
		IsReady := "1"
	}
	if IsObject(Value)
	{
		return ObjGetCapacity(Value) != "" ? (Value.HasKey("__Class") ? "Class" : "Object")
		: IsFunc(Value) ? "Func"
		: ComObjType(Value) != "" ? "ComObject"
		: NumGet(&Value, 0, "Ptr") == nBoundFunc ? "BoundFunc"
		: NumGet(&Value, 0, "Ptr") == nMatchObj ? "RegExMatchInfo"
		: NumGet(&Value, 0, "Ptr") == nFileObj ? "FileObject"
		: NumGet(&Value, 0, "Ptr") == nEnumObj ? "Object::Enumerator"
		: "Property"
	}
	else if (ObjGetCapacity([Value], 1) != "")
		return "String"
	else
		return InStr(Value, ".") ? "Float" : "Integer"
}
VarSetStrCapacity(ByRef TargetVar, RequestedCapacity:="") ;ByRef by necessity
{
	static ChrSize, IsReady
	if !VarSetCapacity(IsReady)
	{
		ChrSize := A_IsUnicode ? 2 : 1
		IsReady := "1"
	}
	if (RequestedCapacity == "")
		return VarSetCapacity(TargetVar) // ChrSize
	else if (RequestedCapacity == -1)
		return VarSetCapacity(TargetVar, -1) // ChrSize
	return VarSetCapacity(TargetVar, RequestedCapacity*ChrSize) // ChrSize
}
WinActivate(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinActivate
}
WinActivateBottom(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hWnd
	if hWnd := AHKFC_WinExistSingleAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		WinActivate, % "ahk_id " hWnd
	else
		WinActivateBottom, % WinTitle, % WinText, % ExcludeTitle, % ExcludeText
}
WinClose(WinTitle:="", WinText:="", SecondsToWait:="", ExcludeTitle:="", ExcludeText:="")
{
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinClose,,, % SecondsToWait
}
WinGetClass(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local OutputVar
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinGetClass, OutputVar
	return OutputVar
}
WinGetClientPos(ByRef X:="", ByRef Y:="", ByRef Width:="", ByRef Height:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hWnd, RECT
	hWnd := AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hWnd
		throw Exception("Target window not found.", -1)

	;GetClientRect function (winuser.h) | Microsoft Docs
	;https://docs.microsoft.com/en-us/windows/win32/api/winuser/nf-winuser-getclientrect
	;The right and bottom members contain the width and height of the window.

	VarSetCapacity(RECT, 16, 0)
	DllCall("user32\GetClientRect", "Ptr",hWnd, "Ptr",&RECT)
	DllCall("user32\ClientToScreen", "Ptr",hWnd, "Ptr",&RECT)
	X := NumGet(&RECT, 0, "Int")
	Y := NumGet(&RECT, 4, "Int")
	Width := NumGet(&RECT, 8, "Int")
	Height := NumGet(&RECT, 12, "Int")
	;note: no return value, returns values via the ByRef variables
}
WinGetControls(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local OutputVar
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinGet, OutputVar, % "ControlList"
	return StrSplit(OutputVar, "`n")
}
WinGetControlsHwnd(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local ControlsHwnd, i, OutputVar
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinGet, OutputVar, % "ControlListHwnd"
	ControlsHwnd := StrSplit(OutputVar, "`n")
	for i in ControlsHwnd
		ControlsHwnd[i] += 0
	return ControlsHwnd
}
WinGetCount(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local OutputVar
	if AHKFC_WinExistSingleAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		return 0 + 1
	WinGet, OutputVar, % "Count", % WinTitle, % WinText, % ExcludeTitle, % ExcludeText
	return 0 + OutputVar
}
WinGetExStyle(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local OutputVar
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinGet, OutputVar, % "ExStyle"
	return 0 + OutputVar
}
WinGetID(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hWnd
	hWnd := AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hWnd
		throw Exception("Target window not found.", -1)
	return 0 + hWnd
}
WinGetIDLast(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local OutputVar
	if OutputVar := AHKFC_WinExistSingleAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		return 0 + OutputVar
	WinGet, OutputVar, % "IDLast", % WinTitle, % WinText, % ExcludeTitle, % ExcludeText
	return 0 + OutputVar
}
WinGetList(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hWnd, List, OutputVar
	if hWnd := AHKFC_WinExistSingleAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		return [hWnd]
	;note: WinGet generates multiple variables of the form: 'OutputVar%A_Index%':
	WinGet, OutputVar, % "List", % WinTitle, % WinText, % ExcludeTitle, % ExcludeText
	List := []
	List.SetCapacity(OutputVar)
	Loop % OutputVar
		List.Push(0+OutputVar%A_Index%)
	return List
}
WinGetMinMax(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local OutputVar
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinGet, OutputVar, % "MinMax"
	if (OutputVar == "")
		return ""
	return 0 + OutputVar
}
WinGetPID(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local OutputVar
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinGet, OutputVar, % "PID"
	if (OutputVar == "")
		return ""
	return 0 + OutputVar
}
WinGetPos(ByRef X:="", ByRef Y:="", ByRef Width:="", ByRef Height:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	X := Y := Width := Height := ""
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinGetPos, X, Y, Width, Height
	;note: no return value, returns values via the ByRef variables
}
WinGetProcessName(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local OutputVar
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinGet, OutputVar, % "ProcessName"
	return OutputVar
}
WinGetProcessPath(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local OutputVar
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinGet, OutputVar, % "ProcessPath"
	return OutputVar
}
WinGetStyle(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local OutputVar
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinGet, OutputVar, % "Style"
	return 0 + OutputVar
}
WinGetText(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local OutputVar
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinGetText, OutputVar
	if ErrorLevel
		throw Exception("", -1)
	return OutputVar
}
WinGetTitle(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local OutputVar
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinGetTitle, OutputVar
	return OutputVar
}
WinGetTransColor(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local OutputVar
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinGet, OutputVar, % "TransColor"
	return OutputVar
}
WinGetTransparent(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local OutputVar
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinGet, OutputVar, % "Transparent"
	if (OutputVar == "")
		return ""
	return 0 + OutputVar
}
WinHide(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinHide
}
WinKill(WinTitle:="", WinText:="", SecondsToWait:="", ExcludeTitle:="", ExcludeText:="")
{
	local PID
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	;WinKill,,, % SecondsToWait

	;bug fix:
	WinClose,,, % SecondsToWait
	if WinExist()
	{
		WinGet, PID, % "PID"
		Process, % "Close", % PID
	}
}
WinMaximize(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinMaximize
}
WinMinimize(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinMinimize
}
WinMinimizeAll()
{
	WinMinimizeAll
}
WinMinimizeAllUndo()
{
	WinMinimizeAllUndo
}
WinMove(Params*) ;X, Y [, Width, Height, WinTitle, WinText, ExcludeTitle, ExcludeText]
{
	local ExcludeText, ExcludeTitle, Height, Width, WinText, WinTitle, X, Y
	local Len
	if (Params.Length() > 8)
		throw Exception("Too many parameters passed to function.", -1)
	if (Len := Params.Length())
	{
		if (Len > 2)
		{
			X := Params[1]
			Y := Params[2]
			Width := Params[3]
			Height := Params[4]
			WinTitle := Params[5]
			WinText := Params[6]
			ExcludeTitle := Params[7]
			ExcludeText := Params[8]
			if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
				throw Exception("Target window not found.", -1)
			WinMove,,, % X, % Y, % Width, % Height
		}
		else
		{
			X := Params[1]
			Y := Params[2]
			WinMove, % X, % y
		}
	}
	else
		WinMove
}
WinMoveBottom(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinSet, % "Bottom"
}
WinMoveTop(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinSet, % "Top"
}
WinRedraw(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinSet, % "Redraw"
}
WinRestore(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinRestore
}
WinSetAlwaysOnTop(Value:=-1, WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hWnd
	if Value in % "On,Off,Toggle"
		throw Exception("Parameter #1 is invalid.", -1)
	if Value in % "1,0,-1" ;On,Off,Toggle
		Value := Value == -1 ? "Toggle" : Value ? "On" : "Off"
	;if Value not in % "On,Off,Toggle"
	;	throw Exception("Parameter #1 is invalid.", -1)
	if !hWnd := AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinSet, % "AlwaysOnTop", % Value, % "ahk_id " hWnd
}
WinSetEnabled(Value, WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hWnd, Style
	if Value in % "On,Off,Toggle"
		throw Exception("Parameter #1 is invalid.", -1)
	if !hWnd := AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)

	;1, 0 and -1 are compared as strings, non-integer values (e.g. 1.0) are not allowed
	if (Value == -1)
	{
		WinGet, Style, % "Style", % "ahk_id " hWnd
		Value := (Style & 0x8000000) ? "On" : "Off" ;WS_DISABLED := 0x8000000
	}

	if (Value == 1)
		WinSet, % "Enable",, % "ahk_id " hWnd
	else if (Value == 0)
		WinSet, % "Disable",, % "ahk_id " hWnd
	else
		throw Exception("Parameter #1 is invalid.", -1)
}
WinSetExStyle(Value, WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinSet, % "ExStyle", % Value
	if ErrorLevel
		throw Exception("", -1)
}
WinSetRegion(Options:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinSet, % "Region", % Options
	if ErrorLevel
		throw Exception("", -1)
}
WinSetStyle(Value, WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local TempStyle ;bug fix
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinSet, % "Style", % Value

	;==============================
	;bug fix:
	if ErrorLevel
	{
		WinGet, TempStyle, % "Style"
		Value := Abs(Value)
		if ((SubStr(Value, 1, 1) = "+") && (TempStyle & Value = Value))
		|| ((SubStr(Value, 1, 1) = "-") && (TempStyle & Value = 0))
		|| (!RegExMatch(Value, "^[+\-^]") && (TempStyle = Value))
			ErrorLevel := 0
	}
	;==============================

	if ErrorLevel
		throw Exception("", -1)
}
WinSetTitle(NewTitle, Params*) ;NewTitle [, WinTitle, WinText, ExcludeTitle, ExcludeText]
{
	local ExcludeText, ExcludeTitle, WinText, WinTitle
	if (Params.Length() > 4)
		throw Exception("Too many parameters passed to function.", -1)
	if (Params.Length())
	{
		WinTitle := Params[1]
		WinText := Params[2]
		ExcludeTitle := Params[3]
		ExcludeText := Params[4]
		if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
			throw Exception("Target window not found.", -1)
		WinSetTitle,,, % NewTitle
	}
	else
		WinSetTitle, % NewTitle
}
WinSetTransColor(Color, WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hWnd
	if (Color == "")
		Color := "Off"
	if !hWnd := AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinSet, % "TransColor", % Color
}
WinSetTransparent(N, WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hWnd
	if (N == "")
		N := "Off"
	if !hWnd := AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinSet, % "Transparent", % N, % "ahk_id " hWnd
}
WinShow(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local DHW
	DHW := A_DetectHiddenWindows
	DetectHiddenWindows, % "On"
	if !AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		throw Exception("Target window not found.", -1)
	WinShow
	DetectHiddenWindows, % DHW
}
WinWait(WinTitle:="", WinText:="", Timeout:="", ExcludeTitle:="", ExcludeText:="")
{
	local hWnd
	if (Timeout == 0)
		Timeout := 0.001
	if hWnd := AHKFC_WinExistSingleAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		return 0 + hWnd
	WinWait, % WinTitle, % WinText, % Timeout, % ExcludeTitle, % ExcludeText
	return ErrorLevel ? 0 : WinExist()
}
WinWaitActive(WinTitle:="", WinText:="", Timeout:="", ExcludeTitle:="", ExcludeText:="")
{
	local hWnd
	if (Timeout == 0)
		Timeout := 0.001
	if hWnd := AHKFC_WinExistSingleAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		WinWaitActive, % "ahk_id " hWnd,, % Timeout
	else
		WinWaitActive, % WinTitle, % WinText, % Timeout, % ExcludeTitle, % ExcludeText
	return ErrorLevel ? 0 : WinExist()
}
WinWaitClose(WinTitle:="", WinText:="", Timeout:="", ExcludeTitle:="", ExcludeText:="")
{
	local hWnd
	if (Timeout == 0)
		Timeout := 0.001
	if hWnd := AHKFC_WinExistSingleAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		WinWaitClose, % "ahk_id " hWnd,, % Timeout
	else
		WinWaitClose, % WinTitle, % WinText, % Timeout, % ExcludeTitle, % ExcludeText
	if ErrorLevel
		AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText) ;update Last Found Window
	return !ErrorLevel
}
WinWaitNotActive(WinTitle:="", WinText:="", Timeout:="", ExcludeTitle:="", ExcludeText:="")
{
	local hWnd
	if (Timeout == 0)
		Timeout := 0.001
	if hWnd := AHKFC_WinExistSingleAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
		WinWaitNotActive, % "ahk_id " hWnd,, % Timeout
	else
		WinWaitNotActive, % WinTitle, % WinText, % Timeout, % ExcludeTitle, % ExcludeText
	return !ErrorLevel
}

;==================================================

;FORMER AHK V2 FUNCTIONS:

;note: maintained here, but with a '1' suffix:

Input1(Options:="", EndKeys:="", MatchList:="")
{
	local OutputVar
	Input, OutputVar, % Options, % EndKeys, % MatchList
	return OutputVar
}
InputEnd1()
{
	;note: this doesn't fully match the (former) AHK v2 function
	Input
	return !ErrorLevel
}
SoundGet1(ComponentType:="", ControlType:="", DeviceNumber:="")
{
	local OutputVar
	SoundGet, OutputVar, % ComponentType, % ControlType, % DeviceNumber
	if OutputVar in % "On,Off"
		OutputVar := (OutputVar = "On")
	if !ErrorLevel
		return OutputVar
}
SoundSet1(NewSetting, ComponentType:="", ControlType:="", DeviceNumber:="")
{
	SoundSet, % NewSetting, % ComponentType, % ControlType, % DeviceNumber
}
StringCaseSense1(OnOffLocale)
{
	if OnOffLocale in % "1,0"
		OnOffLocale := OnOffLocale ? "On" : "Off"
	StringCaseSense, % OnOffLocale
}

;versions of ControlXXX functions that use window coordinates (whereas AHK v2 uses client coordinates)
;these functions are halfway houses, so you can convert the function now, but adjust the X/Y coordinates later

;ControlClick
ControlGetPos1(ByRef X:="", ByRef Y:="", ByRef Width:="", ByRef Height:="", Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlGetPos, X, Y, Width, Height,, % "ahk_id " hCtl
	;note: AHK v1: does not set ErrorLevel
	;if ErrorLevel
	;	throw Exception("", -1)
	;note: no return value, returns values via the ByRef variables
}
ControlMove1(X:="", Y:="", Width:="", Height:="", Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl
	hCtl := AHKFC_ControlGetHwndAux(Control, WinTitle, WinText, ExcludeTitle, ExcludeText)
	if !hCtl
		throw Exception("Target control not found.", -1)
	ControlMove,, % X, % Y, % Width, % Height, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
}

;==================================================

;AUXILIARY FUNCTIONS (WINDOWS/CONTROLS):

;note: should be essentially the same as ControlGetHwnd, but should not throw
AHKFC_ControlGetHwndAux(Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hCtl
	if Control is % "integer"
	{
		if (Type(Control) == "Integer")
			return 0 + WinExist("ahk_id " Control)
	}
	if IsObject(Control)
	{
		if !Control.Hwnd
			throw Exception("", -2)
		return 0 + WinExist("ahk_id " Control.Hwnd)
	}
	if (Control == "")
		return 0 + AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
	try ControlGet, hCtl, % "Hwnd",, % Control, % "ahk_id " AHKFC_WinExistAux(WinTitle, WinText, ExcludeTitle, ExcludeText)
	catch
		throw Exception("", -2)
	return 0 + hCtl
}

AHKFC_WinActiveAux(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	if WinTitle is % "integer"
	{
		if (Type(WinTitle) == "Integer")
			return 0 + WinActive("ahk_id " WinTitle)
	}
	if IsObject(WinTitle)
	{
		if !WinTitle.Hwnd
			throw Exception("", -2)
		return 0 + WinActive("ahk_id " WinTitle.Hwnd)
	}
	return 0 + WinActive(WinTitle, WinText, ExcludeTitle, ExcludeText)
}

AHKFC_WinExistAux(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hWnd
	if WinTitle is % "integer"
	{
		if (Type(WinTitle) == "Integer")
			return 0 + WinExist("ahk_id " WinTitle)
	}
	if IsObject(WinTitle)
	{
		if !WinTitle.Hwnd
			throw Exception("", -2)
		return 0 + WinExist("ahk_id " WinTitle.Hwnd)
	}
	if (hWnd := WinExist(WinTitle, WinText, ExcludeTitle, ExcludeText))
		return 0 + hWnd

	;custom functionality: if WinExist("A") fails when the window is switching,
	;try multiple times to get the active window:
	if (WinTitle = "A")
	{
		Loop 60 ;3000 ms/50 ms = 60
		{
			if (hWnd := WinExist(WinTitle, WinText, ExcludeTitle, ExcludeText))
				break
			Sleep, 50
			;SoundBeep, 1000 ;diagnostic
			;SoundBeep, 1000 ;diagnostic
		}
	}
	return 0 + hWnd
}

AHKFC_WinExistSingleAux(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	;note: WinText/ExcludeTitle/ExcludeText are unused
	if WinTitle is % "integer"
	{
		if (Type(WinTitle) == "Integer")
			return 0 + WinExist("ahk_id " WinTitle)
	}
	if IsObject(WinTitle)
	{
		if !WinTitle.Hwnd
			throw Exception("", -2)
		return 0 + WinExist("ahk_id " WinTitle.Hwnd)
	}
}

;note: similar to the AHK source code's GetNonChildParent
AHKFC_GetNonChildParent(hCtl)
{
	local hWnd, hWndParent, hWndTemp
	hWndTemp := hCtl
	Loop
	{
		;GWL_STYLE := -16 ;WS_CHILD := 0x40000000
		;if not a child window or has no parent
		if !(DllCall("user32\GetWindowLong" (A_PtrSize=8?"Ptr":""), "Ptr",hWndTemp, "Int",-16, "Ptr") & 0x40000000)
		|| !(hWndParent := DllCall("user32\GetParent", "Ptr",hWndTemp, "Ptr"))
		{
			hWnd := hWndTemp
			break
		}
		hWndTemp := hWndParent
	}
	return hWnd
}

;==================================================

;AUXILIARY FUNCTIONS (SORT):

AHKFC_SortAux(Arg1, Arg2, Offset)
{
	global zAHKFC_SortFunc ;note: use 'z' so it appears low down on ListVars
	return %zAHKFC_SortFunc%(Arg1, Arg2, Offset)
}

AHKFC_SortLogical(Arg1, Arg2, Offset)
{
	local Ret
	Ret := DllCall("shlwapi\StrCmpLogicalW", "WStr",Arg1, "WStr",Arg2)
	return Ret ? Ret : -Offset
}

;==================================================

;CALLBACKCREATE AND CALLBACKFREE:

;based on:
;AHK-misc./callbackcreate.ahk at master · HelgeffegleH/AHK-misc. · GitHub
;https://github.com/HelgeffegleH/AHK-misc./blob/master/functions/callbackcreate/callbackcreate.ahk

CallbackCreate(Function, Options:="", ParamCount:="")
{
	; Address := CallbackCreate(Function, Options := "", ParamCount := Function.MinParams)
	; see: https://lexikos.github.io/v2/docs/commands/CallbackCreate.htm
	; Note that the v2 version doesn't affect A_EventInfo, this version does.
	; /*
	; This function fails when:
	; Function is not an object or a valid function name.
	; Function has a MinParams property which exceeds the number of parameters that the callback will supply.
	; ParamCount is negative.
	; ParamCount is omitted and: 1) Function has no MinParams property; or 2) the & option is used with the standard 32-bit calling convention.
	; */

	local actual_param_count, cbo, fn, has_minparams, isobj, minparams, pass_params_pointer, pcb, require_param_count, use_cdecl
	static IsReady, router_fn
	if !VarSetCapacity(IsReady)
	{
		router_fn := "AHKFC_CallbackCreateRouter"
		IsReady := "1"
	}
	; Input validation start. This is partly 'ported' from BIF_CallbackCreate (a097)
	if (!isobj := IsObject(Function))
	; && !IsFunc(Function) ; uncomment this line to allow function name strings
		throw Exception("Parameter #1 is invalid.", -1)

	fn := "" ; If a function name was passed, retrieve the function reference below.
	minparams := (isobj ? Function : (fn := Func(Function))).MinParams

	if (minparams != "")
	{
		if minparams is not % "number" ; if the MinParams property doesn't return a number, minparams is 0.
			minparams := 0
	}
	has_minparams := (minparams != "") ? True : False
	minparams := has_minparams ? minparams : 0

	use_cdecl := InStr(options, "C")

	require_param_count := (A_PtrSize == 4) && !use_cdecl
	pass_params_pointer := InStr(options, "&", True)

	if (ParamCount != "")
	{
		actual_param_count := ParamCount
		if ((actual_param_count < 0)
		|| (has_minparams && (pass_params_pointer ? 1 : actual_param_count) < minparams))
			throw Exception("Parameter #3 is invalid.", -1)
	}
	else if (!has_minparams || (pass_params_pointer && require_param_count))
		throw Exception("Parameter #3 must not be blank in this case.", -1)
	else
		actual_param_count := minparams
	if (A_PtrSize == 4 && (!use_cdecl && actual_param_count > 31))
		throw Exception("Parameter #3 is invalid.", -1)
	; Input validation end.

	Function := isobj ? Function : fn ; Make sure Function is a function reference and not just a name, avoids finding the reference on each callback.

	; Callback object, its address is passed to the callback router via A_EventInfo
	cbo := {actual_param_count:actual_param_count
	, pass_params_pointer:pass_params_pointer
	, Function:Function}

	if (fn && !pass_params_pointer && !fn.IsVariadic)
	{
		; Meaning the name of a non variadic function was passed as Function, and not using the '&' option.
		; No need to route the callback.
		pcb := RegisterCallback(fn.name, options, actual_param_count)
	}
	else
	{
		; A Function object or name of a variadic function or '&' option was passed. Needs to route the callback
		pcb := RegisterCallback(router_fn, options, actual_param_count, &cbo)
	}
	if !pcb ; Hopefully, this shouldn't happen
		throw Exception("RegisterCallback failed", -1) ;[FIXME] original code lacked '-1' (code suggestion)

	ObjAddRef(&cbo) ; This is decremented in CallbackFree when freeing the callback.
	CallbackFree(cbo, pcb) ; Add to cache. pcb is for internal use only.
	return pcb ; Return the callback address
}

CallbackFree(Address, pcb:="") ;note: pcb is for internal use only
{
	; Address, the address to free.
	; pcb, internal use, always omit. Used for setting/getting callback functions
	; see: https://lexikos.github.io/v2/docs/commands/CallbackCreate.htm
	static callback_cache, IsReady
	if !VarSetCapacity(IsReady)
	{
		callback_cache := [] ; Contains all callback objects.
		IsReady := "1"
	}
	if pcb ; add to cache
	{
		if IsObject(Address)
			return callback_cache[pcb] := Address
		throw Exception("?", -1) ;[FIXME] original code lacked '-1' (code suggestion)
	}
	if address is not % "number"
		address := 0
	if (address < 65536 && address >= 0)
		throw Exception("Parameter #1 is invalid.", -1)
	; Free the address and callback object.
	DllCall("kernel32\GlobalFree", "Ptr",Address, "Ptr")
	ObjRelease(&callback_cache.delete(Address))
	return ;[FIXME] if necessary, add comment, else remove (code suggestion)
}

AHKFC_CallbackCreateRouter(p*)
{
	; Help function.
	; Routes all callbacks.
	; Since this function is variadic, all parameters are passed by address.
	static IsReady, p_type
	local args, cbo
	if !VarSetCapacity(IsReady)
	{
		p_type := (A_PtrSize == 4) ? "UInt" : "Int64"
		IsReady := "1"
	}
	cbo := Object(A_EventInfo) ; A_EventInfo contains the address of the callback object.
	if !cbo.pass_params_pointer ; The '&' option was not used, fetch all parameters and pass directly to the script callback.
	{
		args := []
		Loop % cbo.actual_param_count
			args.Push(NumGet(p+0, A_PtrSize*(A_Index-1), p_type))
	}
	else ; using the "&" option, just pass the pointer to the arguments to the script callback.
		args := [p]
	return cbo.function.call(args*) ; Call script callback
}

;==================================================

;CLASSES: A_TRAYMENU:

;class A_TrayMenu
;{
;	static varClickCount := 2
;	ClickCount[]
;	{
;		get
;		{
;			return this.varClickCount
;		}
;		set
;		{
;			Menu, Tray, Click, % value
;			return this.varClickCount := value
;		}
;	}
;}

;note: for A_TrayMenu.ClickCount
class A_TrayMenu ;extends AHKMenuClass (from 'JEEAhk1FCGui.ahk')
{
	static varClickCount := 2
	static varMenuName := "Tray"
	;static base := {__Call:"AHKMenuClass.__Call"}
	;static base := {Add:"AHKMenuClass.Add"}

	__New()
	{
		local varnameA_TrayMenu
		;global AHKMenuClass ;causes error: 'Error: Global variables must not be declared in this function.'
		;'static' makes AHK v1 run this on script startup
		static init := ""
		static dummy := new A_TrayMenu
		if init
			return init
		;A_TrayMenu.base := {__Set:"MyStaticSingletonClass.__Set"}
		;MsgBox, % IsFunc("AHKMenuClassExists")
		;if VarSetCapacity(AHKMenuClassExists)
		;&& IsObject(AHKMenuClass)
		if IsFunc("AHKMenuClassExists")
		{
			;A_TrayMenu := new AHKMenuClass
			;A_TrayMenu.base := AHKMenuClass.base
			varnameA_TrayMenu := "A_TrayMenu" ;pick an unusual name to avoid 'This local variable has the same name as a global variable.' error
			%varnameA_TrayMenu% := new AHKMenuClass
			A_TrayMenu.varMenuName := "Tray"
		}
		else
			return init
	}

	ClickCount[]
	{
		get
		{
			return this.varClickCount
		}
		set
		{
			Menu, % "Tray", % "Click", % value
			return this.varClickCount := value
		}
	}
	ZAdd(MenuItemName:="", CallbackOrSubmenu:="", Options:="")
	{
		;MsgBox, % this.varMenuName ;diagnostic
		if IsObject(CallbackOrSubmenu)
			Menu, % this.varMenuName, % "Add", % MenuItemName, % ":" CallbackOrSubmenu.varMenuName, % Options
		else if (CallbackOrSubmenu == "")
			Menu, % this.varMenuName, % "Add"
		else if IsFunc(CallbackOrSubmenu)
			Menu, % this.varMenuName, % "Add", % MenuItemName, % CallbackOrSubmenu, % Options
	}
	ZDelete(Params*)
	{
		if Params.HasKey(1)
			Menu, % this.varMenuName, % "Delete", % Params[1]
		else if !Params.Length()
		{
			Menu, % this.varMenuName, % "NoStandard"
			Menu, % this.varMenuName, % "DeleteAll"
		}
	}
}

;==================================================

;CLASSES: ARRAY:

;Array Object - Methods & Properties | AutoHotkey v2
;https://lexikos.github.io/v2/docs/objects/Array.htm

;note: this will change [] and Array(), (but not StrSplit's return value)
Array(Params*)
{
	global Array
	;ObjSetBase(Params, Array)
	static FuncObjSetBase, IsReady
	if !VarSetCapacity(IsReady)
	{
		FuncObjSetBase := Func("ObjSetBase")
		IsReady := "1"
	}
	if FuncObjSetBase
		%FuncObjSetBase%(Params, Array)
	return Params
}

class Array
{
	;Has(vKey)
	;{
	;	return ObjHasKey(this, vKey)
	;}
	Length[]
	{
		get
		{
			return ObjLength(this)
		}
		set
		{
		}
	}
	Count[]
	{
		get
		{
			return ObjCount(this)
		}
		set
		{
		}
	}
	Sort(vOpt:="", vFunc:="") ;provisional Sort method in case it ever gets implemented in AHK v2
	{
		local oTemp
		if StrLen(vOpt)
			throw Exception("Options parameter not supported.", -1)
		oTemp := BIFEx_ArrSort(this, vOpt, vFunc)
		this.RemoveAt(1, this.Length)
		this.InsertAt(1, oTemp*)
	}
}

;==================================================

;CLASSES: BUFFER:

class Buffer
{
	__New(Size, FillByte:="")
	{
		this.SetCapacity("_Buf", Size)
		this._Size := Size
		if (FillByte != "")
			DllCall("kernel32\RtlFillMemory", "Ptr",this.GetAddress("_Buf"), "UPtr",Size, "UChar",FillByte)
		;note: use ObjRawSet to set key values too, to support AHK v1 C++ code I wrote that checks for object keys:
		ObjRawSet(this, "Size", Size)
		ObjRawSet(this, "Ptr", this.GetAddress("_Buf"))
	}
	Size[]
	{
		get
		{
			return this._Size
		}
		set
		{
			if (value != this._Size)
				this.SetCapacity("_Buf", value)
			;if (value > this._Size)
			;	DllCall("kernel32\RtlFillMemory", "Ptr",this.GetAddress("_Buf")+this._Size, "UPtr",value-this._Size, "UChar",0)
			;note: use ObjRawSet to set key values too, to support AHK v1 C++ code I wrote that checks for object keys:
			ObjRawSet(this, "Size", value)
			return this._Size := value
		}
	}
	Ptr[]
	{
		get
		{
			return this.GetAddress("_Buf")
		}
		set
		{
			throw Exception("Invalid usage.", -1)
		}
	}
	;note: Buffer.Data does not handle nulls as AHK v2 does
	;Data[]
	;{
	;	get
	;	{
	;		return StrGet(this.GetAddress("_Buf"), this._Size/2, "UTF-16")
	;	}
	;	set
	;	{
	;		throw Exception("Invalid usage.", -1)
	;	}
	;}
	__Get(Property, Params*)
	{
		if (Property != "Ptr")
		&& (Property != "Size")
		&& (Property != "__Class")
			throw Exception("Unknown property.", -1)
	}
	__Set(Property, Params*)
	{
		if (Property != "_Size")
		&& (Property != "Ptr")
		&& (Property != "Size")
		&& (Property != "__Class")
			throw Exception("Unknown property.", -1)
	}
	__Call(Method, Params*)
	{
		if (Method != "GetAddress")
		&& (Method != "SetCapacity")
			throw Exception("Unknown method.", -1)
	}
}

;==================================================

;CLASSES: CLIPBOARDALL:

;we use Buffer, in this library, instead of a ClipboardAll class,
;because this would crash AutoHotkey v1:
;class ClipboardAll extends Buffer
;{
;}

;==================================================

;CLASSES: MAP:

Map(Params*)
{
	if (Params.Length() & 1)
		throw Exception("Invalid number of parameters.", -1)
	return Object(Params*)
}

;note: potential function for creating a map slightly more similar to the AHK v2 map:
;Map(Params*)
;{
;	global
;	local Map2
;	if (Params.Length() & 1)
;		throw Exception("Invalid number of parameters.", -1)
;	Map2 := Object(Params*)
;	ObjSetBase(Map2, Map)
;	return Map2
;}

;class Map
;{
;	;Has(vKey)
;	;{
;	;	return ObjHasKey(this, vKey)
;	;}
;	Count[]
;	{
;		get
;		{
;			;return ObjCount(this)
;			static FuncObjCount := Func("ObjCount")
;			if FuncObjCount
;				return %FuncObjCount%(this)
;			else
;				throw Exception("", -1)
;		}
;		set
;		{
;		}
;	}
;}

;==================================================

;FUNCTIONS: COMCALL:

;note: old version, without ByRef handling:
;ComCall(Index, ComObj, Params*)
;{
;	local Ptr, Result
;	Ptr := IsObject(ComObj) ? ComObj.Ptr : ComObj
;	Result := DllCall(NumGet(NumGet(Ptr+0, "Ptr")+Index*A_PtrSize, "Ptr"), "Ptr",Ptr, Params*)
;	if !(Params.Length() & 1) ;if even number (i.e. no return type)
;	&& (Result < 0)
;		throw Exception("", -1, Result)
;	return Result
;}

;note: handles up to 20 type/argument pairs
;note: 'ComCall(Index, ComObj)' (only 2 parameters) is valid e.g. ITaskbarList::HrInit
ComCall(Index, ComObj, v1:="`f`a`b",ByRef v2:="`f`a`b", v3:="`f`a`b",ByRef v4:="`f`a`b", v5:="`f`a`b",ByRef v6:="`f`a`b", v7:="`f`a`b",ByRef v8:="`f`a`b", v9:="`f`a`b",ByRef v10:="`f`a`b", v11:="`f`a`b",ByRef v12:="`f`a`b", v13:="`f`a`b",ByRef v14:="`f`a`b", v15:="`f`a`b",ByRef v16:="`f`a`b", v17:="`f`a`b",ByRef v18:="`f`a`b", v19:="`f`a`b",ByRef v20:="`f`a`b", v21:="`f`a`b",ByRef v22:="`f`a`b", v23:="`f`a`b",ByRef v24:="`f`a`b", v25:="`f`a`b",ByRef v26:="`f`a`b", v27:="`f`a`b",ByRef v28:="`f`a`b", v29:="`f`a`b",ByRef v30:="`f`a`b", v31:="`f`a`b",ByRef v32:="`f`a`b", v33:="`f`a`b",ByRef v34:="`f`a`b", v35:="`f`a`b",ByRef v36:="`f`a`b", v37:="`f`a`b",ByRef v38:="`f`a`b", v39:="`f`a`b",ByRef v40:="`f`a`b", v41:="`f`a`b") ;ByRef because ComCall/DllCall sometimes modify variables
{
	local Count, Err, IsHResult, pFunc, pIfc, Result

	;==============================
	;note: custom code to add a 'disable throw' option to ComCall:
	;(this is necessary for 'BIFEx_ComCall' that helps recreate the AHK source code (C++) as AHK code)
	;(it uses 'BIFEx_FAILED', equivalent to the 'FAILED' macro, instead)
	local AllowThrow := 1
	if (Ord(Index) == 95) ;for internal use only with BIFEx_ComCall (prefix Index with '_' to suppress throwing)
	{
		Index := SubStr(Index, 2)
		AllowThrow := 0
	}
	;==============================

	static Default, FuncIsSet, IsReady
	if !VarSetCapacity(IsReady)
	{
		Default := "`f`a`b" ;arbitrary default value (equivalent to 'unset' in AHK v2):
		FuncIsSet := Func("IsSet")
		IsReady := "1"
	}

	if Index is not % "number"
		throw Exception("Parameter #1 is invalid.", -1)

	pIfc := IsObject(ComObj) ? ComObjValue(ComObj) : ComObj
	;pIfc := IsObject(ComObj) ? ComObject(ComObj) : ComObj ;[FIXME] better? worse? than above (see v2-changes.htm: ComObjFromPtr)

	;find last argument:
	Count := 41
	Loop % Count
	{
		if (v%Count% != Default)
			break
		Count--
	}

	;check arguments are valid:
	Loop % Count
	{
		;if IsSet is backported to AHK v1.1, it can check if the parameters have a value:
		if FuncIsSet && !%FuncIsSet%(v%A_Index%)
			throw Exception("This variable has not been assigned a value.", -1) ;note: unlike AHK v2, this doesn't state the variable name
		if (v%A_Index% == Default)
			throw Exception(A_Index & 1 ? "Invalid arg type." : "Missing a required parameter.", -1)
	}

	IsHResult := 0
	if (Count & 1)
	{
		if InStr(v%Count%, "HRESULT")
		{
			v%Count% := InStr(v%Count%, "Cdecl") ? "Cdecl" : "Int"
			IsHResult := 1
		}
	}
	else
	{
		Count++
		v%Count% := "Int"
		IsHResult := 1
	}

	pFunc := NumGet(NumGet(pIfc+0, 0, "Ptr")+Index*A_PtrSize, 0, "Ptr")

	if (Count == 1)
		Result := DllCall(pFunc, "Ptr",pIfc, v1)
	else if (Count == 3)
		Result := DllCall(pFunc, "Ptr",pIfc, v1,v2, v3)
	else if (Count == 5)
		Result := DllCall(pFunc, "Ptr",pIfc, v1,v2, v3,v4, v5)
	else if (Count == 7)
		Result := DllCall(pFunc, "Ptr",pIfc, v1,v2, v3,v4, v5,v6, v7)
	else if (Count == 9)
		Result := DllCall(pFunc, "Ptr",pIfc, v1,v2, v3,v4, v5,v6, v7,v8, v9)
	else if (Count == 11)
		Result := DllCall(pFunc, "Ptr",pIfc, v1,v2, v3,v4, v5,v6, v7,v8, v9,v10, v11)
	else if (Count == 13)
		Result := DllCall(pFunc, "Ptr",pIfc, v1,v2, v3,v4, v5,v6, v7,v8, v9,v10, v11,v12, v13)
	else if (Count == 15)
		Result := DllCall(pFunc, "Ptr",pIfc, v1,v2, v3,v4, v5,v6, v7,v8, v9,v10, v11,v12, v13,v14, v15)
	else if (Count == 17)
		Result := DllCall(pFunc, "Ptr",pIfc, v1,v2, v3,v4, v5,v6, v7,v8, v9,v10, v11,v12, v13,v14, v15,v16, v17)
	else if (Count == 19)
		Result := DllCall(pFunc, "Ptr",pIfc, v1,v2, v3,v4, v5,v6, v7,v8, v9,v10, v11,v12, v13,v14, v15,v16, v17,v18, v19)
	else if (Count == 21)
		Result := DllCall(pFunc, "Ptr",pIfc, v1,v2, v3,v4, v5,v6, v7,v8, v9,v10, v11,v12, v13,v14, v15,v16, v17,v18, v19,v20, v21)
	else if (Count == 23)
		Result := DllCall(pFunc, "Ptr",pIfc, v1,v2, v3,v4, v5,v6, v7,v8, v9,v10, v11,v12, v13,v14, v15,v16, v17,v18, v19,v20, v21,v22, v23)
	else if (Count == 25)
		Result := DllCall(pFunc, "Ptr",pIfc, v1,v2, v3,v4, v5,v6, v7,v8, v9,v10, v11,v12, v13,v14, v15,v16, v17,v18, v19,v20, v21,v22, v23,v24, v25)
	else if (Count == 27)
		Result := DllCall(pFunc, "Ptr",pIfc, v1,v2, v3,v4, v5,v6, v7,v8, v9,v10, v11,v12, v13,v14, v15,v16, v17,v18, v19,v20, v21,v22, v23,v24, v25,v26, v27)
	else if (Count == 29)
		Result := DllCall(pFunc, "Ptr",pIfc, v1,v2, v3,v4, v5,v6, v7,v8, v9,v10, v11,v12, v13,v14, v15,v16, v17,v18, v19,v20, v21,v22, v23,v24, v25,v26, v27,v28, v29)
	else if (Count == 31)
		Result := DllCall(pFunc, "Ptr",pIfc, v1,v2, v3,v4, v5,v6, v7,v8, v9,v10, v11,v12, v13,v14, v15,v16, v17,v18, v19,v20, v21,v22, v23,v24, v25,v26, v27,v28, v29,v30, v31)
	else if (Count == 33)
		Result := DllCall(pFunc, "Ptr",pIfc, v1,v2, v3,v4, v5,v6, v7,v8, v9,v10, v11,v12, v13,v14, v15,v16, v17,v18, v19,v20, v21,v22, v23,v24, v25,v26, v27,v28, v29,v30, v31,v32, v33)
	else if (Count == 35)
		Result := DllCall(pFunc, "Ptr",pIfc, v1,v2, v3,v4, v5,v6, v7,v8, v9,v10, v11,v12, v13,v14, v15,v16, v17,v18, v19,v20, v21,v22, v23,v24, v25,v26, v27,v28, v29,v30, v31,v32, v33,v34, v35)
	else if (Count == 37)
		Result := DllCall(pFunc, "Ptr",pIfc, v1,v2, v3,v4, v5,v6, v7,v8, v9,v10, v11,v12, v13,v14, v15,v16, v17,v18, v19,v20, v21,v22, v23,v24, v25,v26, v27,v28, v29,v30, v31,v32, v33,v34, v35,v36, v37)
	else if (Count == 39)
		Result := DllCall(pFunc, "Ptr",pIfc, v1,v2, v3,v4, v5,v6, v7,v8, v9,v10, v11,v12, v13,v14, v15,v16, v17,v18, v19,v20, v21,v22, v23,v24, v25,v26, v27,v28, v29,v30, v31,v32, v33,v34, v35,v36, v37,v38, v39)
	else if (Count == 41)
		Result := DllCall(pFunc, "Ptr",pIfc, v1,v2, v3,v4, v5,v6, v7,v8, v9,v10, v11,v12, v13,v14, v15,v16, v17,v18, v19,v20, v21,v22, v23,v24, v25,v26, v27,v28, v29,v30, v31,v32, v33,v34, v35,v36, v37,v38, v39,v40, v41)

	if !AllowThrow
		return Result

	if IsHResult && (Result < 0)
		throw Exception("HResult: " Result, -1, Result) ;[FIXME] as far as possible, match AHK v2's ComCall error messages (and Exception.Extra value)
	if (ErrorLevel == 0)
		return Result
	else
		throw Exception(ErrorLevel, -1) ;[FIXME] as far as possible, match AHK v2's ComCall error messages (it would help this lib, and users generally, if the ComCall/DllCall/Error documentation pages gave more exact details, and concrete error message examples) (wish list idea)
}

;==================================================

;SOURCE CODE AUXILIARY FUNCTIONS:

;note: BIFEx_ComCall exists to help recreate AutoHotkey built-in functions (BIFs) as AHK code,
;it works like (the custom backport of) ComCall, however it does not throw,
;functions that use BIFEx_ComCall should check its return value with the BIFEx_FAILED function (based on the FAILED macro),
;the function uses try/catch in case something unexpected happens in the ComCall custom backport
BIFEx_ComCall(Index, ComObj, v1:="`f`a`b",ByRef v2:="`f`a`b", v3:="`f`a`b",ByRef v4:="`f`a`b", v5:="`f`a`b",ByRef v6:="`f`a`b", v7:="`f`a`b",ByRef v8:="`f`a`b", v9:="`f`a`b",ByRef v10:="`f`a`b", v11:="`f`a`b",ByRef v12:="`f`a`b", v13:="`f`a`b",ByRef v14:="`f`a`b", v15:="`f`a`b",ByRef v16:="`f`a`b", v17:="`f`a`b",ByRef v18:="`f`a`b", v19:="`f`a`b",ByRef v20:="`f`a`b", v21:="`f`a`b",ByRef v22:="`f`a`b", v23:="`f`a`b",ByRef v24:="`f`a`b", v25:="`f`a`b",ByRef v26:="`f`a`b", v27:="`f`a`b",ByRef v28:="`f`a`b", v29:="`f`a`b",ByRef v30:="`f`a`b", v31:="`f`a`b",ByRef v32:="`f`a`b", v33:="`f`a`b",ByRef v34:="`f`a`b", v35:="`f`a`b",ByRef v36:="`f`a`b", v37:="`f`a`b",ByRef v38:="`f`a`b", v39:="`f`a`b",ByRef v40:="`f`a`b", v41:="`f`a`b") ;for internal use only
{
	local vRet
	;note: ComCall custom backport: for internal use only: prefix Index with '_' to suppress throwing:
	try vRet := ComCall("_" Index, ComObj, v1,v2, v3,v4, v5,v6, v7,v8, v9,v10, v11,v12, v13,v14, v15,v16, v17,v18, v19,v20, v21,v22, v23,v24, v25,v26, v27,v28, v29,v30, v31,v32, v33,v34, v35,v36, v37,v38, v39,v40, v41)
	catch
		throw Exception("BIFEx_ComCall error.", -1)
	return vRet
}

;note: this should exactly match BIFEx_IsHex (BIFEx_IsHex is stored in a separate file)
AHKFC_IsHex(vNum)
{
	return RegExMatch(vNum, "^0[xX][\dA-Fa-f]$")
}

;based on AHK v2's IsNumeric() (util.cpp) (very similar to AHK v1's IsPureNumeric() (util.cpp))
;note: this should exactly match BIFEx_IsNumeric (BIFEx_IsNumeric is stored in a separate file)
AHKFC_IsNumeric(vNum, vAllowNegative:=0, vAllowAllWhitespace:=1, vAllowFloat:=0, vAllowImpure:=0)
{
	local vChar, vDoSkip, vHasAtLeastOneDigit, vHasDecimalPoint, vHasExponent, vIsHex, vNextChar, vText
	static PURE_NOT_NUMERIC := 0
	static PURE_INTEGER := 1
	static PURE_FLOAT := 2

	vNum := Trim(vNum) ;note: this applies LTrim and RTrim, source code only applies LTrim
	if !StrLen(vNum)
		return vAllowAllWhitespace ? PURE_INTEGER : PURE_NOT_NUMERIC

	if (SubStr(vNum, 1, 1) = "-")
	{
		if (vAllowNegative)
			vNum := SubStr(vNum, 2)
		else
			return PURE_NOT_NUMERIC
	}
	else if (SubStr(vNum, 1, 1) = "+")
		vNum := SubStr(vNum, 2)

	vIsHex := AHKFC_IsHex(vNum)
	if vIsHex
		vNum := SubStr(vNum, 3)

	vHasDecimalPoint := False
	vHasExponent := False
	vHasAtLeastOneDigit := False ;i.e. a string consisting of only "+", "-" or "." is not considered numeric.

	vDoSkip := 0
	Loop Parse, vNum
	{
		if vDoSkip
		{
			vDoSkip := 0
			continue
		}
		vChar := A_LoopField
		if RegExMatch(vChar, "[ `t]")
		{
			if StrLen(LTrim(SubStr(vNum, A_Index)))
			&& !vAllowImpure
				return PURE_NOT_NUMERIC
			break
		}
		if (vChar == ".")
		{
			;note: 2 decimal points not allowed
			;note: decimal points in hex not allowed
			if (!vAllowFloat || vHasDecimalPoint || vIsHex) ;if vAllowFloat==False, a decimal point at the very end of the number is considered non-numeric
				return PURE_NOT_NUMERIC
			else
				vHasDecimalPoint := True
		}
		else
		{
			;if hex: if not a hex digit
			;if not hex: if not a digit
			if (vIsHex ? !RegExMatch(vChar, "[\dA-fa-f]") : !RegExMatch(vChar, "\d"))
			{
				if (vAllowImpure)
				{
					if (vHasAtLeastOneDigit)
						return vHasDecimalPoint ? PURE_FLOAT : PURE_INTEGER
					else ;i.e. the strings "." and "-" are not considered to be numeric by themselves.
						return PURE_NOT_NUMERIC
				}
				else
				{
					if (StrUpper(vChar) != "E")
					|| !(vHasDecimalPoint && vHasAtLeastOneDigit)
						return PURE_NOT_NUMERIC
					;AHK v2 replacement:
					;if (StrUpper(vChar) != "E")
					;|| !vHasAtLeastOneDigit
					;|| vHasExponent
					;|| !vAllowFloat
					;	return PURE_NOT_NUMERIC

					;if reaches here, char is 'E'/'e'
					vNextChar := SubStr(vNum, A_Index+1, 1)
					if RegExMatch(vNextChar, "[\-+]") ;The optional sign is present on the exponent.
					{
						vDoSkip := 1 ;Omit it from further consideration so that the outer loop doesn't see it as an extra/illegal sign.
						vNextChar := SubStr(vNum, A_Index+2, 1)
					}
					if RegExMatch(vNextChar, "\D")
					|| (vNextChar == "")
						;Even if it is an 'e', ensure what follows it is a valid exponent
						return PURE_NOT_NUMERIC

					;AHK v2 addition:
					;vHasExponent := True
					;vHasDecimalPoint := True ;For simplicity, since a decimal point after the exponent isn't valid.
				}
			}
			else ;a digit or hex-digit
				vHasAtLeastOneDigit := True
		}
	}

	if (vHasAtLeastOneDigit)
		return vHasDecimalPoint ? PURE_FLOAT : PURE_INTEGER
	else
		return PURE_NOT_NUMERIC
}

;SUCCEEDED macro (winerror.h) - Win32 apps | Microsoft Docs
;https://docs.microsoft.com/en-us/windows/win32/api/winerror/nf-winerror-succeeded
;FAILED macro (winerror.h) - Win32 apps | Microsoft Docs
;https://docs.microsoft.com/en-us/windows/win32/api/winerror/nf-winerror-failed

BIFEx_SUCCEEDED(vNum)
{
	return (vNum >= 0)
}

BIFEx_FAILED(vNum)
{
	return (vNum < 0)
}

BIFEx_PropertyKeyCreate(vGUID, vPropertyID)
{
	local oPropertyKey
	oPropertyKey := Buffer(20, 0)
	DllCall("ole32\CLSIDFromString", "WStr",vGUID, "Ptr",oPropertyKey.Ptr) ;fmtid
	NumPut(vPropertyID, oPropertyKey.Ptr, 16, "UInt") ;pid
	return oPropertyKey
}

;==================================================

;SOURCE CODE AUXILIARY FUNCTIONS (FOR SOUNDXXX FUNCTIONS):

;note: unlike the AHK source code function: this returns a string and closes CoTaskMemFree,
;the AHK source code function returns a pointer to inside a PROPVARIANT struct
BIFEx_SoundDeviceGetName(pMMDevice)
{
	local hResult, oPropertyKey, oPROPVARIANT, pStore, vAddr, vDevName
	static vIsV1 := (InStr(A_AhkVersion, "1.") == 1)
	pStore := 0
	oPROPVARIANT := Buffer(A_PtrSize=8?24:16, 0)
	;STGM_READ := 0x0
	hResult := BIFEx_ComCall(4, pMMDevice, "UInt",0x0, "Ptr*",vIsV1?pStore:&pStore) ;IMMDevice::OpenPropertyStore
	if BIFEx_SUCCEEDED(hResult)
	{
		oPropertyKey := BIFEx_PropertyKeyCreate("{A45C254E-DF1C-4EFD-8020-67D146A850E0}", 14) ;PKEY_Device_FriendlyName ;source: functiondiscoverykeys_devpkey.h
		hResult := BIFEx_ComCall(5, pStore, "Ptr",oPropertyKey.Ptr, "Ptr",oPROPVARIANT.Ptr) ;IPropertyStore::GetValue
		if BIFEx_FAILED(hResult)
			return ""
		vAddr := NumGet(oPROPVARIANT.Ptr, 8, "Ptr") ;pwszVal
		vDevName := StrGet(vAddr, "UTF-16")
		DllCall("ole32\CoTaskMemFree", "Ptr",vAddr)
		ObjRelease(pStore)
		return vDevName
	}
	return ""
}

BIFEx_SoundSetGet_GetDevice(vDeviceStr, ByRef pMMDevice)
{
	local hResult, pMMDeviceCollection, pMMDeviceEnumerator, pMMDeviceTemp, vDelimPos, vDevName, vTargetIndex, vTargetName, vTargetNameLen, vZBIndex
	static vIsV1 := (InStr(A_AhkVersion, "1.") == 1)
	static CLSCTX_ALL := 0x17 ;source: combaseapi.h
	pMMDeviceEnumerator := 0
	pMMDeviceCollection := 0
	hResult := 0
	pMMDevice := 0

	static CLSID_MMDeviceEnumerator := "{BCDE0395-E52F-467C-8E3D-C4579291692E}" ;source: mmdeviceapi.h
	static IID_IMMDeviceEnumerator := "{A95664D2-9614-4F35-A746-DE8DB63617E6}" ;source: mmdeviceapi.h
	static oCLSID, oIID, vIsReady := 0
	if !vIsReady
	{
		oCLSID := Buffer(16, 0)
		DllCall("ole32\CLSIDFromString", "WStr",CLSID_MMDeviceEnumerator, "Ptr",oCLSID.Ptr)
		oIID := Buffer(16, 0)
		DllCall("ole32\CLSIDFromString", "WStr",IID_IMMDeviceEnumerator, "Ptr",oIID.Ptr)
		vIsReady := 1
	}

	hResult := DllCall("ole32\CoCreateInstance", "Ptr",oCLSID.Ptr, "Ptr",0, "UInt",CLSCTX_ALL, "Ptr",oIID.Ptr, "Ptr*",vIsV1?pMMDeviceEnumerator:&pMMDeviceEnumerator)
	if BIFEx_SUCCEEDED(hResult)
	{
		if (vDeviceStr == "")
		{
			;Get default playback device.
			;eRender := 0, eConsole := 0 ;source: mmdeviceapi.h
			hResult := BIFEx_ComCall(4, pMMDeviceEnumerator, "Int",0, "Int",0, "Ptr*",vIsV1?pMMDevice:&pMMDevice) ;IMMDeviceEnumerator::GetDefaultAudioEndpoint
		}
		else
		{
			;Parse Name:Index, Name or Index.
			vTargetIndex := 0
			vTargetName := ""
			static vIsV1X := InStr(1, 1,, 0)
			vDelimPos := InStr(vDeviceStr, ":", 0, vIsV1X-1)
			if (vDelimPos)
			{
				vTargetIndex := SubStr(vDeviceStr, vDelimPos+1)
				vTargetName := SubStr(vDeviceStr, 1, vDelimPos-1)
			}
			else if (AHKFC_IsNumeric(vDeviceStr, False, False))
				vTargetIndex := vDeviceStr - 1
			else
				vTargetName := vDeviceStr
			vTargetNameLen := StrLen(vTargetName)

			;Enumerate devices: include unplugged devices so that indices don't change when a device is plugged in.
			;eAll := 2 ;source: mmdeviceapi.h
			;DEVICE_STATE_ACTIVE := 0x1 ;source: mmdeviceapi.h
			;DEVICE_STATE_UNPLUGGED := 0x8 ;source: mmdeviceapi.h
			hResult := BIFEx_ComCall(3, pMMDeviceEnumerator, "Int",2, "UInt",0x9, "Ptr*",vIsV1?pMMDeviceCollection:&pMMDeviceCollection) ;IMMDeviceEnumerator::EnumAudioEndpoints
			if BIFEx_SUCCEEDED(hResult)
			{
				if StrLen(vTargetName)
				{
					pMMDeviceTemp := 0
					Loop
					{
						vZBIndex := A_Index - 1
						hResult := BIFEx_ComCall(4, pMMDeviceCollection, "UInt",vZBIndex, "Ptr*",vIsV1?pMMDeviceTemp:&pMMDeviceTemp) ;IMMDeviceCollection::Item
						if BIFEx_FAILED(hResult)
							break
						if (vDevName := BIFEx_SoundDeviceGetName(pMMDeviceTemp))
						{
							if (SubStr(vDevName, 1, vTargetNameLen) = vTargetName)
							&& (vTargetIndex-- == 0)
							{
								pMMDevice := pMMDeviceTemp
								break
							}
						}
						ObjRelease(pMMDeviceTemp)
					}
				}
				else
					hResult := BIFEx_ComCall(4, pMMDeviceCollection, "UInt",vTargetIndex, "Ptr*",vIsV1?pMMDevice:&pMMDevice) ;IMMDeviceCollection::Item
				ObjRelease(pMMDeviceCollection)
			}
		}
		ObjRelease(pMMDeviceEnumerator)
	}
	return hResult
}

;note: the source code has 2 functions with the same name: 'SoundSetGet_FindComponent',
;this library adds 2 suffixes to differentiate between them: 'Part' and 'Device'
;note: pRoot, like pPart, is a pointer to an IPart interface
BIFEx_SoundSetGet_FindComponentPart(pRoot, oSearch)
{
	local hResult, i, pPart, pParts, vCheckName, vPartCount, vPartName, vPartType, vTemp
	static vIsV1 := (InStr(A_AhkVersion, "1.") == 1)
	static CLSCTX_ALL := 0x17 ;source: combaseapi.h
	pParts := 0 ;IPartsList
	pPart := 0 ;IPart
	vPartCount := 0
	vPartType := 0
	vPartName := ""

	;source code version of line below: bool check_name = *oSearch.target_name
	vCheckName := !!StrLen(oSearch.target_name)

	;eRender := 0, eCapture := 1 ;source: mmdeviceapi.h
	if (oSearch.data_flow == 0) ;In := 0 ;source: devicetopology.h
		hResult := BIFEx_ComCall(10, pRoot, "Ptr*",vIsV1?pParts:&pParts) ;IPart::EnumPartsIncoming
	else
		hResult := BIFEx_ComCall(11, pRoot, "Ptr*",vIsV1?pParts:&pParts) ;IPart::EnumPartsOutgoing

	if BIFEx_FAILED(hResult)
		return 0

	if BIFEx_FAILED(BIFEx_ComCall(3, pParts, "Int*",vIsV1?vPartCount:&vPartCount)) ;IPartsList::GetCount
		vPartCount := 0

	Loop % vPartCount
	{
		i := A_Index - 1
		if BIFEx_FAILED(BIFEx_ComCall(4, pParts, "UInt",i, "Ptr*",vIsV1?pPart:&pPart)) ;IPartsList::GetPart
			continue

		if BIFEx_SUCCEEDED(BIFEx_ComCall(6, pPart, "Int*",vIsV1?vPartType:&vPartType)) ;IPart::GetPartType
		{
			if (vPartType == 0) ;Connector := 0 ;source: devicetopology.h
			{
				if ((vPartCount == 1) ;Ignore Connectors with no Subunits of their own.
				&& (!vCheckName || (BIFEx_SUCCEEDED(BIFEx_ComCall(3, pPart, "Ptr*",vIsV1?vPartName:&vPartName)) && (vPartName = oSearch.target_name)))) ;IPart::GetName
				{
					if (++oSearch.count == oSearch.target_instance)
					{
						vTemp := 0
						if (oSearch.target_control == 3) ;SoundControlType::IID: ;arbitrary AHK source code constant
						{
							;Permit retrieving the IPart or IConnector itself.  Since there may be
							;multiple connected Subunits (and they can be enumerated or retrieved
							;via the Connector IPart), this is only done for the Connector.
							BIFEx_ComCall(0, pPart, "Ptr",oSearch.target_iid.Ptr, "Ptr*",vIsV1?vTemp:&vTemp) ;IUnknown::QueryInterface
							oSearch.control := vTemp
						}
						else if (oSearch.target_control == 2) ;SoundControlType::Name: ;arbitrary AHK source code constant
						{
							;check_name would typically be false in this case, since a value of true
							;would mean the caller already knows the component's name.
							BIFEx_ComCall(3, pPart, "Ptr*",vIsV1?vTemp:&vTemp) ;IPart::GetName
							oSearch.name := vTemp
						}
						ObjRelease(pPart)
						ObjRelease(pParts)
						return True
					}
				}
			}
			else ;Subunit
			{
				;Recursively find the Connector nodes linked to this part.
				if (BIFEx_SoundSetGet_FindComponentPart(pPart, oSearch))
				{
					;A matching connector part has been found with this part as one of the nodes used
					;to reach it.  Therefore, if this part supports the requested control interface,
					;it can in theory be used to control the component.  An example path might be:
					;Output < Master Mute < Master Volume < Sum < Mute < Volume < CD Audio
					;Parts are considered from right to left, as we return from recursion.
					if (!oSearch.control && !oSearch.ignore_remaining_subunits)
					{
						;Query this part for the requested interface and let caller check the result.
						vTemp := 0
						BIFEx_ComCall(13, pPart, "UInt",CLSCTX_ALL, "Ptr",oSearch.target_iid.Ptr, "Ptr*",vIsV1?vTemp:&vTemp) ;IPart::Activate
						oSearch.control := vTemp

						;If this subunit has siblings, ignore any controls further up the line
						;as they're likely shared by other components (i.e. master controls).
						if (vPartCount > 1)
							oSearch.ignore_remaining_subunits := True
					}
					ObjRelease(pPart)
					ObjRelease(pParts)
					return True
				}
			}
		}
		ObjRelease(pPart)
	}
	ObjRelease(pParts)
	return False
}

;note: the source code has 2 functions with the same name: 'SoundSetGet_FindComponent',
;this library adds 2 suffixes to differentiate between them: 'Part' and 'Device'
BIFEx_SoundSetGet_FindComponentDevice(pMMDevice, oSearch)
{
	local oIID, oIID2, pConn, pConnTo, pPart, pTopo, vTemp
	static vIsV1 := (InStr(A_AhkVersion, "1.") == 1)
	static CLSCTX_ALL := 0x17 ;source: combaseapi.h
	pTopo := 0 ;IDeviceTopology
	pConn := pConnTo := 0 ;IConnector
	pPart := 0 ;IPart

	oSearch.count := 0
	oSearch.control := 0
	oSearch.name := 0
	oSearch.ignore_remaining_subunits := 0

	oIID := Buffer(16, 0)
	static IID_IDeviceTopology := "{2A07407E-6497-4A18-9787-32F79BD0D98F}" ;source: devicetopology.h
	DllCall("ole32\CLSIDFromString", "WStr",IID_IDeviceTopology, "Ptr",oIID.Ptr)
	oIID2 := Buffer(16, 0)
	static IID_IPart := "{AE2DE0E4-5BCA-4F2D-AA46-5D13F8FDB3A9}" ;source: devicetopology.h
	DllCall("ole32\CLSIDFromString", "WStr",IID_IPart, "Ptr",oIID2.Ptr)

	if BIFEx_SUCCEEDED(BIFEx_ComCall(3, pMMDevice, "Ptr",oIID.Ptr, "UInt",CLSCTX_ALL, "Ptr",0, "Ptr*",vIsV1?pTopo:&pTopo)) ;IMMDevice::Activate
	{
		if BIFEx_SUCCEEDED(BIFEx_ComCall(4, pTopo, "UInt",0, "Ptr*",vIsV1?pConn:&pConn)) ;IDeviceTopology::GetConnector
		{
			vTemp := 0
			;eRender := 0, eCapture := 1 ;source: mmdeviceapi.h
			if BIFEx_SUCCEEDED(BIFEx_ComCall(4, pConn, "Ptr*",vIsV1?vTemp:&vTemp)) ;IConnector::GetDataFlow
			&& ((oSearch.data_flow := vTemp) || 1)
			&& BIFEx_SUCCEEDED(BIFEx_ComCall(8, pConn, "Ptr*",vIsV1?pConnTo:&pConnTo)) ;IConnector::GetConnectedTo
			{
				;note: QueryInterface with only 1 parameter in the source code,
				;perhaps it calls this:
				;IUnknown::QueryInterface(Q,) - Win32 apps | Microsoft Docs
				;https://docs.microsoft.com/en-us/windows/win32/api/unknwn/nf-unknwn-iunknown-queryinterface(q)
				;A helper function template that infers an interface identifier, and calls QueryInterface(REFIID,void).
				if BIFEx_SUCCEEDED(BIFEx_ComCall(0, pConnTo, "Ptr",oIID2.Ptr, "Ptr*",vIsV1?pPart:&pPart)) ;IUnknown::QueryInterface
				{
					;Search: the result is stored in the search struct.
					BIFEx_SoundSetGet_FindComponentPart(pPart, oSearch)
					ObjRelease(pPart)
				}
				ObjRelease(pConnTo)
			}
			ObjRelease(pConn)
		}
		ObjRelease(pTopo)
	}
	return (oSearch.count == oSearch.target_instance)
}

BIFEx_SoundConvertComponent(vDeviceStr, oSearch)
{
	local vDelimPos
	static PURE_INTEGER := 1
	if (AHKFC_IsNumeric(vDeviceStr) == PURE_INTEGER)
	{
		oSearch.target_name := ""
		oSearch.target_instance := vDeviceStr
	}
	else
	{
		static vIsV1 := InStr(10, 1,, -1, 1)
		vDelimPos := InStr(vDeviceStr, ":", 0, vIsV1-1)
		if (vDelimPos)
		{
			oSearch.target_instance := SubStr(vDeviceStr, vDelimPos+1)
			oSearch.target_name := SubStr(vDeviceStr, 1, vDelimPos-1)
		}
		else
			oSearch.target_name := vDeviceStr
	}
}

BIFEx_Sound(vFunc, vSetting:="pseudonull", vComponent:="", vDevice:="")
{
	local hResult, oBufFloat, oIID, oLevel, oLevelMin, oLevelRange, oSearch, pAudioEndpointVolume, pAudioMute, pAudioVolumeLevel, pIfc, pMMDevice, vAddr, vAdjustCurrentSetting, vBool, vChannelCount, vDB, vDevName, vErr, vIID, vMaxDB, vMaxLevel, vMinDB, vRet, vSettingScalar, vSoundControlType, vSoundModeIsSet, vStepping, vTemp, vZBIndex
	static vIsV1 := (InStr(A_AhkVersion, "1.") == 1)
	static CLSCTX_ALL := 0x17 ;source: combaseapi.h
	;arbitrary values in AHK source code:
	if InStr(vFunc, "Volume")
		vSoundControlType := 0
	else if InStr(vFunc, "Mute")
		vSoundControlType := 1
	else if InStr(vFunc, "Name")
		vSoundControlType := 2
	else if InStr(vFunc, "Interface") ;IID
		vSoundControlType := 3
	static ERR_PARAM1_INVALID := "Parameter #1 is invalid."

	static IID_IAudioVolumeLevel := "{7FB7B48F-531D-44A2-BCB3-5AD5A134B3DC}" ;source: devicetopology.h
	static IID_IAudioMute := "{DF45AEEA-B74A-4B6B-AFAD-2366B6AA012E}" ;source: devicetopology.h
	static IID_IAudioEndpointVolume := "{5CDF2C82-841E-4546-9722-0CF74078229A}" ;source: endpointvolume.h

	oSearch := {}
	oSearch.target_control := vSoundControlType

	if (oSearch.target_control == 0) ;Volume
	{
		oIID := Buffer(16, 0)
		hResult := DllCall("ole32\CLSIDFromString", "WStr",IID_IAudioVolumeLevel, "Ptr",oIID.Ptr)
		oSearch.target_iid := oIID
	}
	else if (oSearch.target_control == 1) ;Mute
	{
		oIID := Buffer(16, 0)
		hResult := DllCall("ole32\CLSIDFromString", "WStr",IID_IAudioMute, "Ptr",oIID.Ptr)
		oSearch.target_iid := oIID
	}
	else if (oSearch.target_control == 3) ;IID
	{
		vIID := vSetting
		if (SubStr(vIID, 1, 1) != "{")
			throw Exception(ERR_PARAM1_INVALID "" (vIID != "" ? "`n`n" "Specifically: " vIID : ""), -1)
		oIID := Buffer(16, 0)
		hResult := DllCall("ole32\CLSIDFromString", "WStr",vIID, "Ptr",oIID.Ptr)
		if BIFEx_FAILED(hResult)
			throw Exception(ERR_PARAM1_INVALID "" (vIID != "" ? "`n`n" "Specifically: " vIID : ""), -1)
		oSearch.target_iid := oIID
	}

	vSoundModeIsSet := (vSetting != "pseudonull") ;Boolean: i.e. if it's not NULL, the mode is "SET".
	vRet := "" ;Set default.

	vSettingScalar := 0
	if (vSoundModeIsSet)
	{
		vSettingScalar := vSetting / 100
		if (vSettingScalar < -1)
			vSettingScalar := -1
		else if (vSettingScalar > 1)
			vSettingScalar := 1
	}

	vBool := vAdjustCurrentSetting := vSetting && RegExMatch(vSetting, "^[+\-]")

	pMMDevice := 0
	hResult := BIFEx_SoundSetGet_GetDevice(vDevice, pMMDevice)
	if BIFEx_FAILED(hResult)
	{
		ErrorLevel := "Device Not Found"
		return
	}

	vErr := ""

	pIfc := 0
	if (vComponent == "") ;Component is Master (omitted).
	{
		if (oSearch.target_control == 3) ;IID
		{
			hResult := BIFEx_ComCall(0, pMMDevice, "Ptr",oSearch.target_iid.Ptr, "Ptr*",vIsV1?pIfc:&pIfc) ;IUnknown::QueryInterface
			if BIFEx_FAILED(hResult)
				hResult := BIFEx_ComCall(3, pMMDevice, "Ptr",oSearch.target_iid.Ptr, "UInt",CLSCTX_ALL, "UInt",0, "Ptr*",vIsV1?pIfc:&pIfc) ;IMMDevice::Activate
			vRet := pIfc
		}
		else if (oSearch.target_control == 2) ;Name
		{
			if (vDevName := BIFEx_SoundDeviceGetName(pMMDevice))
			{
				vRet := vDevName
				;DllCall("ole32\CoTaskMemFree", "Ptr",&PROPVARIANT+8) ;needed in source code, but not here, as custom version of BIFEx_SoundDeviceGetName does CoTaskMemFree
			}
		}
		else ;Volume / Mute
		{
			;For Master/Speakers, use the IAudioEndpointVolume interface.  Some devices support master
			;volume control, but do not actually have a volume subunit (so the other method would fail).
			pAudioEndpointVolume := 0
			oIID := Buffer(16, 0)
			DllCall("ole32\CLSIDFromString", "WStr",IID_IAudioEndpointVolume, "Ptr",oIID.Ptr)
			hResult := BIFEx_ComCall(3, pMMDevice, "Ptr",oIID.Ptr, "UInt",CLSCTX_ALL, "UInt",0, "Ptr*",vIsV1?pAudioEndpointVolume:&pAudioEndpointVolume) ;IMMDevice::Activate
			if (BIFEx_SUCCEEDED(hResult))
			{
				if (oSearch.target_control == 0) ;Volume
				{
					if (!vSoundModeIsSet || vAdjustCurrentSetting)
					{
						hResult := BIFEx_ComCall(9, pAudioEndpointVolume, "Float*",vIsV1?vRet:&vRet) ;IAudioEndpointVolume::GetMasterVolumeLevelScalar
					}

					if (BIFEx_SUCCEEDED(hResult))
					{
						if (vSoundModeIsSet)
						{
							if (vAdjustCurrentSetting)
							{
								vSettingScalar += vRet
								if (vSettingScalar > 1)
									vSettingScalar := 1
								else if (vSettingScalar < 0)
									vSettingScalar := 0
							}
							hResult := BIFEx_ComCall(7, pAudioEndpointVolume, "Float",vSettingScalar, "Ptr",0) ;IAudioEndpointVolume::SetMasterVolumeLevelScalar
						}
						else
						{
							vRet *= 100.0
						}
					}
				}
				else ;Mute.
				{
					if (!vSoundModeIsSet || vAdjustCurrentSetting)
					{
						BIFEx_ComCall(15, pAudioEndpointVolume, "Int*",vIsV1?vRet:&vRet) ;IAudioEndpointVolume::GetMute
					}
					if (vSoundModeIsSet && BIFEx_SUCCEEDED(hResult))
					{
						BIFEx_ComCall(14, pAudioEndpointVolume, "Int",vAdjustCurrentSetting ? !vRet : (vSettingScalar > 0), "Ptr",0) ;IAudioEndpointVolume::SetMute
					}
				}
				ObjRelease(pAudioEndpointVolume)
			}
		}
	}
	else
	{
		BIFEx_SoundConvertComponent(vComponent, oSearch)

		if (!BIFEx_SoundSetGet_FindComponentDevice(pMMDevice, oSearch))
		{
			vErr := "Component Not Found"
		}
		else if (oSearch.target_control == 3) ;IID
		{
			vRet := oSearch.control ;pointer to interface
			oSearch.control := 0 ;Don't release it.
		}
		else if (oSearch.target_control == 2) ;Name
		{
			vRet := oSearch.name
		}
		else if (!oSearch.control)
		{
			vErr := "Component Doesn't Support This Control Type"
		}
		else if (oSearch.target_control == 0) ;Volume
		{
			pAudioVolumeLevel := oSearch.control

			vChannelCount := 0
			if BIFEx_FAILED(BIFEx_ComCall(3, pAudioVolumeLevel, "UInt*",vIsV1?vChannelCount:&vChannelCount)) ;IAudioVolumeLevel::GetChannelCount
				goto BIFEx_Sound_control_fail

			oLevel := []
			(vIsV1) ? oLevel.SetCapacity(vChannelCount) : oLevel.Capacity := vChannelCount
			oLevelMin := []
			(vIsV1) ? oLevelMin.SetCapacity(vChannelCount) : oLevelMin.Capacity := vChannelCount
			oLevelRange := []
			(vIsV1) ? oLevelRange.SetCapacity(vChannelCount) : oLevelRange.Capacity := vChannelCount
			vStepping := vDB := vMinDB := vMaxDB := 0
			vMaxLevel := 0

			Loop % vChannelCount
			{
				vZBIndex := A_Index - 1
				if (BIFEx_FAILED(BIFEx_ComCall(5, pAudioVolumeLevel, "UInt",vZBIndex, "Float*",vIsV1?vDB:&vDB)) ;IAudioVolumeLevel::GetLevel
				|| BIFEx_FAILED(BIFEx_ComCall(4, pAudioVolumeLevel, "UInt",vZBIndex, "Float*",vIsV1?vMinDB:&vMinDB, "Float*",vIsV1?vMaxDB:&vMaxDB, "Float*",vIsV1?vStepping:&vStepping))) ;IAudioVolumeLevel::GetLevelRange
					goto BIFEx_Sound_control_fail
				;Convert dB to scalar.
				oLevelMin[vZBIndex] := 10 ** (vMinDB/20)
				oLevelRange[vZBIndex] := 10**(vMaxDB/20) - oLevelMin[vZBIndex]
				;Compensate for differing level ranges. (No effect if range is -96..0 dB.)
				oLevel[vZBIndex] := (10**(vDB/20)-oLevelMin[vZBIndex]) / oLevelRange[vZBIndex]
				;Windows reports the highest level as the overall volume.
				if (vMaxLevel < oLevel[vZBIndex])
					vMaxLevel := oLevel[vZBIndex]
			}

			if (vSoundModeIsSet)
			{
				if (vAdjustCurrentSetting)
				{
					vSettingScalar += vMaxLevel
					if (vSettingScalar > 1)
						vSettingScalar := 1
					else if (vSettingScalar < 0)
						vSettingScalar := 0
				}

				Loop % vChannelCount
				{
					vZBIndex := A_Index - 1
					vTemp := vSettingScalar
					if (vMaxLevel)
						vTemp *= (oLevel[vZBIndex]/vMaxLevel) ;Preserve balance.
					;Compensate for differing level ranges.
					vTemp := oLevelMin[vZBIndex] + vTemp*oLevelRange[vZBIndex]
					;Convert scalar to dB.
					oLevel[vZBIndex] := 20 * Log(vTemp) ;Log10
				}

				oBufFloat := Buffer(4*3*vChannelCount, 0)
				vAddr := oBufFloat.Ptr
				Loop % vChannelCount
				{
					NumPut(oLevel[A_Index], vAddr+0, 0, "Float")
					NumPut(oLevelMin[A_Index], vAddr+0, 4, "Float")
					NumPut(oLevelRange[A_Index], vAddr+0, 8, "Float")
					vAddr += 12
				}
				hResult := BIFEx_ComCall(8, pAudioVolumeLevel, "Ptr",oBufFloat.Ptr, "UInt",vChannelCount, "Ptr",0) ;IAudioVolumeLevel::SetLevelAllChannels
			}
			else
			{
				vRet := vMaxLevel * 100
			}
		}
		else if (oSearch.target_control == 1) ;Mute
		{
			pAudioMute := oSearch.control

			if (!vSoundModeIsSet || vAdjustCurrentSetting)
			{
				hResult := BIFEx_ComCall(4, pAudioMute, "Int*",vIsV1?vRet:&vRet) ;IAudioMute::GetMute
			}
			if (vSoundModeIsSet && BIFEx_SUCCEEDED(hResult))
			{
				hResult := BIFEx_ComCall(3, pAudioMute, "Int",vAdjustCurrentSetting ? !vRet : (vSettingScalar > 0), "Ptr",0) ;IAudioMute::SetMute
			}
		}
		BIFEx_Sound_control_fail:
		if (oSearch.control)
			ObjRelease(oSearch.control)
		;if (oSearch.name)
		;	DllCall("ole32\CoTaskMemFree", "Ptr",&PROPVARIANT+8) ;needed in source code, but not here, as custom version of BIFEx_SoundDeviceGetName does CoTaskMemFree
	}

	ObjRelease(pMMDevice)

	if BIFEx_FAILED(hResult)
	{
		if (vSoundModeIsSet)
			vErr := "Can't Change Setting"
		else
			vErr := "Can't Get Current Setting"
	}
	if (vErr != "")
		ErrorLevel := vErr
	else
		ErrorLevel := 0

	if (vSoundModeIsSet || ErrorLevel)
		return vRet

	return vRet
}

;==================================================

;BIF: FileSelect
BIF_FileSelect(vOptions:=0, vRootDir_Filename:="", vTitle:="", vFilter:="")
{
	local hWnd, oArrayCOMDLG_FILTERSPEC, oFiles, oIID_IShellItem, pFD, pFOD, pSI, pSIA, pText, vAlwaysUseSaveDialog, vAttrib, vCount, vDefaultFileName, vFilterAll, vFilterCount, vFlags, vGreeting, vIsCLSID, vIsDir, vName, vPath2, vPattern, vPatternAll, vPatternEnd, vPatternStart, vPos1, vPos2, vPosEndBrace, vPosLastBackslash, vResult, vRet, vTChars, vWorkingDir
	static vIsV1 := (InStr(A_AhkVersion, "1.") == 1)
	static MAX_PATH := 260
	static FILE_ATTRIBUTE_DIRECTORY := 0x10 ;source: winnt.h
	static SIGDN_FILESYSPATH := 0x80058000 ;source: ShObjIdl.h
	static IID_IShellItem := "{43826d1e-e718-42ee-bc55-a1e261c37bfe}" ;source: ShObjIdl.h

	static FOS_OVERWRITEPROMPT := 0x2 ;source: ShObjIdl.h
	static FOS_STRICTFILETYPES := 0x4
	static FOS_NOCHANGEDIR := 0x8
	static FOS_PICKFOLDERS := 0x20
	static FOS_FORCEFILESYSTEM := 0x40
	static FOS_ALLNONSTORAGEITEMS := 0x80
	static FOS_NOVALIDATE := 0x100
	static FOS_ALLOWMULTISELECT := 0x200
	static FOS_PATHMUSTEXIST := 0x800
	static FOS_FILEMUSTEXIST := 0x1000
	static FOS_CREATEPROMPT := 0x2000
	static FOS_SHAREAWARE := 0x4000
	static FOS_NOREADONLYRETURN := 0x8000
	static FOS_NOTESTFILECREATE := 0x10000
	static FOS_HIDEMRUPLACES := 0x20000
	static FOS_HIDEPINNEDPLACES := 0x40000
	static FOS_NODEREFERENCELINKS := 0x100000
	static FOS_DONTADDTORECENT := 0x2000000
	static FOS_FORCESHOWHIDDEN := 0x10000000
	static FOS_DEFAULTNOMINIMODE := 0x20000000
	static FOS_FORCEPREVIEWPANEON := 0x40000000
	static FOS_SUPPORTSTREAMABLEITEMS := 0x80000000

	;note: IFileDialog/IFileOpenDialog/IFileSaveDialog methods 0-26 are equivalent
	;static IID_IFileDialog := "{42f85136-db7e-439c-85f1-e4075d135fc8}"
	static CLSID_FileOpenDialog := "{DC1C5A9C-E88A-4dde-A5A1-60F82A20AEF7}"
	static IID_IFileOpenDialog := "{d57c7288-d4ad-4768-be02-9d969532d960}"
	static CLSID_FileSaveDialog := "{C0B4E2F3-BA21-4773-8DBA-335EC946EB8B}"
	static IID_IFileSaveDialog := "{84BCCD23-5FDE-4CDB-AEA4-AF64B83D78AB}"

	static ERR_PARAM4_MUST_BE_BLANK := "Parameter #4 must be blank in this case."

	vDefaultFileName := ""
	if (vRootDir_Filename == "")
		vWorkingDir := ""
	else
	{
		if (StrLen(vRootDir_Filename) >= MAX_PATH)
		{
			vTChars := DllCall("kernel32\GetShortPathName", "Str","\\?\" vRootDir_Filename, "Ptr",0, "UInt",0, "UInt")
			VarSetStrCapacity(vPath2, vTChars)
			if DllCall("kernel32\GetShortPathName", "Str","\\?\" vRootDir_Filename, "Str",vPath2, "UInt",vTChars, "UInt")
				vWorkingDir := (SubStr(vPath2, 1, 4) = "\\?\") ? SubStr(vPath2, 5) : vPath2
		}
		else
			vWorkingDir := vRootDir_Filename
		if (vIsCLSID := BIFEx_StrStarts(vWorkingDir, "::{"))
		{
			if (vPosEndBrace := InStr(vWorkingDir, "}"))
				vIsDir := (vPosEndBrace = StrLen(vWorkingDir)) ;First '}' is also the last char in string, so it's naked CLSID (so assume directory).
				|| BIFEx_StrEnds(vWorkingDir, "\") ;Or path ends in backslash.
			else ;Badly formatted clsid.
				vIsDir := True ;Arbitrary default due to rarity.
		}
		else ;Not a CLSID.
		{
			vAttrib := DllCall("kernel32\GetFileAttributes", "Str",vWorkingDir, "UInt")
			vIsDir := (vAttrib != 0xFFFFFFFF) && (vAttrib & FILE_ATTRIBUTE_DIRECTORY)
		}
		if (!vIsDir)
		{
			;Above condition indicates it's either an existing file that's not a folder, or a nonexistent
			;folder/filename.  In either case, it seems best to assume it's a file because the user may want
			;to provide a default SAVE filename, and it would be normal for such a file not to already exist.
			if (vPosLastBackslash := BIFEx_InStrRev(vWorkingDir, "\"))
			{
				vDefaultFileName := SubStr(vWorkingDir, vPosLastBackslash+1)
				vRootDir_Filename := SubStr(vWorkingDir, 1, vPosLastBackslash-1)
			}
			else ;The entire working_dir string is the default file (unless this is a clsid).
				if (!vIsCLSID)
					vDefaultFileName := vWorkingDir ;This signals it to use the default directory.
			;	else leave working_dir set to the entire clsid string in case it's somehow valid.
		}
		;else it is a directory, so just leave working_dir set as it was initially.
	}

	vPattern := ""
	if StrLen(vFilter)
	{
		vPatternStart := InStr(vFilter, "(")
		if (vPatternStart)
		{
			;Make pattern a separate string because we want to remove any spaces from it.
			;For example, if the user specified Documents (*.txt; *.doc), the space after
			;the semicolon should be removed for the pattern string itself but not from
			;the displayed version of the pattern:
			vPatternEnd := BIFEx_InStrRev(vFilter, ")")
			vPos1 := vPatternStart + 1
			vPos2 := vPatternEnd - 1
			if (vPatternEnd)
				vPattern := SubStr(vFilter, vPos1, vPos2-vPos1+1)
		}
		else ;No open-paren, so assume the entire string is the filter.
			vPattern := vFilter
	}
	vFilterCount := 0
	oArrayCOMDLG_FILTERSPEC := Buffer(2*2*A_PtrSize, 0) ;space for 2 items
	if StrLen(vPattern)
	{
		;Remove any spaces present in the pattern, such as a space after every semicolon
		;that separates the allowed file extensions.  The API docs specify that there
		;should be no spaces in the pattern itself, even though it's okay if they exist
		;in the displayed name of the file-type:
		;Update by Lexikos: Don't remove spaces, since that gives incorrect behaviour for more
		;complex patterns like "prefix *.ext" (where the space should be considered part of the
		;pattern).  Although the docs for OPENFILENAMEW say "Do not include spaces", it may be
		;just because spaces are considered part of the pattern.  On the other hand, the docs
		;relating to IFileDialog::SetFileTypes() say nothing about spaces; and in fact, using a
		;pattern like "*.cpp; *.h" will work correctly (possibly due to how leading spaces work
		;with the file system).
		;StrReplace(pattern, _T(" "), _T(""), SCS_SENSITIVE);

		;note: filter/pattern must be WStr:
		VarSetCapacity(vFilterW, StrLen(vFilter)*2)
		StrPut(vFilter, StrPtr(vFilterW), "UTF-16")
		VarSetCapacity(vPatternW, StrLen(vPattern)*2)
		StrPut(vPattern, StrPtr(vPatternW), "UTF-16")

		NumPut(StrPtr(vFilterW), oArrayCOMDLG_FILTERSPEC.Ptr, 0, "Ptr") ;pszName
		NumPut(StrPtr(vPatternW), oArrayCOMDLG_FILTERSPEC.Ptr, A_PtrSize, "Ptr") ;pszSpec
		vFilterCount++
	}
	;Always include the All Files (*.*) filter, since there doesn't seem to be much
	;point to making this an option.  This is because the user could always type
	;*.* (or *) and press ENTER in the filename field and achieve the same result:
	vFilterAll := "All Files (*.*)"
	vPatternAll := "*.*"

	;note: filter/pattern must be WStr:
	VarSetCapacity(vFilterAllW, StrLen(vFilterAll)*2)
	StrPut(vFilterAll, StrPtr(vFilterAllW), "UTF-16")
	VarSetCapacity(vPatternAllW, StrLen(vPatternAll)*2)
	StrPut(vPatternAll, StrPtr(vPatternAllW), "UTF-16")

	NumPut(StrPtr(vFilterAllW), oArrayCOMDLG_FILTERSPEC.Ptr, vFilterCount*2*A_PtrSize, "Ptr") ;pszName
	NumPut(StrPtr(vPatternAllW), oArrayCOMDLG_FILTERSPEC.Ptr, vFilterCount*2*A_PtrSize+A_PtrSize, "Ptr") ;pszSpec
	vFilterCount++

	;v1.0.43.09: OFN_NODEREFERENCELINKS is now omitted by default because most people probably want a click
	;on a shortcut to navigate to the shortcut's target rather than select the shortcut and end the dialog.
	;v2: Testing on Windows 7 and 10 indicated IFileDialog doesn't change the working directory while the
	;user navigates, unlike GetOpenFileName/GetSaveFileName, and doesn't appear to affect the CWD at all.
	vFlags := FOS_NOCHANGEDIR ;FOS_NOCHANGEDIR according to MS: "Don't change the current working directory."

	vAlwaysUseSaveDialog := False
	if InStr(vOptions, "D")
	{
		vFlags |= FOS_PICKFOLDERS
		if StrLen(vFilter)
			throw Exception(ERR_PARAM4_MUST_BE_BLANK, -1)
		vFilterCount := 0
	}
	if InStr(vOptions, "M")
		vFlags |= FOS_ALLOWMULTISELECT
	if InStr(vOptions, "S") ;Have a "Save" button rather than an "Open" button.
		vAlwaysUseSaveDialog := True

	if StrLen(vTitle)
		vGreeting := vTitle
	else
		vGreeting := Format("Select {:s} - {:s}", (vFlags & FOS_PICKFOLDERS) ? "Folder" : "File", A_ScriptName)

	if (vOptions & 0x20)
		vFlags |= FOS_NODEREFERENCELINKS
	if (vOptions & 0x10)
		vFlags |= FOS_OVERWRITEPROMPT
	if (vOptions & 0x08)
		vFlags |= FOS_CREATEPROMPT
	if (vOptions & 0x02)
		vFlags |= FOS_PATHMUSTEXIST
	if (vOptions & 0x01)
		vFlags |= FOS_FILEMUSTEXIST

	;Despite old documentation indicating it was due to an "OS quirk", previous versions were specifically
	;designed to enable the Save button when OFN_OVERWRITEPROMPT is present but not OFN_CREATEPROMPT, since
	;the former requires the Save dialog while the latter requires the Open dialog.  If both options are
	;present, the caller must specify or omit "S" to choose the dialog type, and one option has no effect.
	if ((vFlags & FOS_OVERWRITEPROMPT) && !(vFlags & (FOS_CREATEPROMPT | FOS_PICKFOLDERS)))
		vAlwaysUseSaveDialog := True

	if vAlwaysUseSaveDialog
	{
		if !(pFD := ComObjCreate(CLSID_FileSaveDialog, IID_IFileSaveDialog))
			return ""
	}
	else
	{
		if !(pFD := ComObjCreate(CLSID_FileOpenDialog, IID_IFileOpenDialog))
			return ""
	}
	;HRESULT hr := CoCreateInstance(vAlwaysUseSaveDialog ? CLSID_FileSaveDialog : CLSID_FileOpenDialog,
	;	NULL, CLSCTX_INPROC_SERVER, IID_PPV_ARGS(&pfd))
	;if (FAILED(hr))
	;	_f_throw_win32(hr)

	BIFEx_ComCall(9, pFD, "UInt",vFlags) ;IFileDialog::SetOptions

	BIFEx_ComCall(17, pFD, "WStr",vGreeting) ;IFileDialog::SetTitle

	if (vFilterCount)
		BIFEx_ComCall(4, pFD, "UInt",vFilterCount, "Ptr",oArrayCOMDLG_FILTERSPEC.Ptr) ;IFileDialog::SetFileTypes

	BIFEx_ComCall(15, pFD, "WStr",vDefaultFileName) ;IFileDialog::SetFileName

	if (StrLen(vWorkingDir) && (vDefaultFileName != vWorkingDir))
	{
		pSI := 0
		oIID_IShellItem := Buffer(16, 0)
		DllCall("ole32\IIDFromString", "WStr",IID_IShellItem, "Ptr",oIID_IShellItem.Ptr)
		if BIFEx_SUCCEEDED(DllCall("shell32\SHCreateItemFromParsingName", "WStr",vWorkingDir, "Ptr",0, "Ptr",oIID_IShellItem.Ptr, "Ptr*",vIsV1?pSI:&pSI))
		{

			BIFEx_ComCall(12, pFD, "Ptr",pSI) ;IFileDialog::SetFolder
			ObjRelease(pSI)
		}
	}

	;At this point, we know a dialog will be displayed.  See macro's comments for details:
	;DIALOG_PREP
	;POST_AHK_DIALOG(0) // Do this only after the above. Must pass 0 for timeout in this case.
	;++g_nFileDialogs
	;auto result := pfd->Show(THREAD_DIALOG_OWNER)
	hWnd := A_ScriptHwnd ;note: hWnd determines icon
	vResult := BIFEx_SUCCEEDED(BIFEx_ComCall(3, pFD, "Ptr",hWnd)) ;IFileDialog::Show
	;--g_nFileDialogs
	;DIALOG_END

	if (vFlags & FOS_ALLOWMULTISELECT)
	{
		oFiles := []
		pFOD := 0
		if BIFEx_SUCCEEDED(vResult)
		&& (pFOD := ComObjQuery(pFD, IID_IFileOpenDialog))
		{
			pSIA := 0
			if BIFEx_SUCCEEDED(BIFEx_ComCall(27, pFOD, "Ptr*",vIsV1?pSIA:&pSIA)) ;IFileOpenDialog::GetResults
			{
				vCount := 0
				BIFEx_ComCall(7, pSIA, "UInt*",vIsV1?vCount:&vCount) ;IShellItemArray::GetCount
				Loop % vCount
				{
					pSI := 0
					if BIFEx_SUCCEEDED(BIFEx_ComCall(8, pSIA, "UInt",A_Index-1, "Ptr*",vIsV1?pSI:&pSI)) ;IShellItemArray::GetItemAt
					{
						pText := 0
						if BIFEx_SUCCEEDED(BIFEx_ComCall(5, pSI, "UInt",SIGDN_FILESYSPATH, "Ptr*",vIsV1?pText:&pText)) ;IShellItem::GetDisplayName
						{
							vName := StrGet(pText, "UTF-16")
							oFiles.Push(vName)
							DllCall("ole32\CoTaskMemFree", "Ptr",pText)
						}
						ObjRelease(pSI)
					}
				}
				ObjRelease(pSIA)
			}
			ObjRelease(pFOD)
		}
		ObjRelease(pFD)
		return oFiles
	}

	;if reach this point: multi-select is off:
	vRet := ""
	pSI := 0
	if BIFEx_SUCCEEDED(vResult)
	&& BIFEx_SUCCEEDED(BIFEx_ComCall(20, pFD, "Ptr*",vIsV1?pSI:&pSI)) ;IFileDialog::GetResult
	{
		pText := 0
		if BIFEx_SUCCEEDED(BIFEx_ComCall(5, pSI, "UInt",SIGDN_FILESYSPATH, "Ptr*",vIsV1?pText:&pText)) ;IShellItem::GetDisplayName
		{
			vRet := StrGet(pText, "UTF-16")
			DllCall("ole32\CoTaskMemFree", "Ptr",pText)
		}
		ObjRelease(pSI)
	}
	;else: User pressed CANCEL vs. OK to dismiss the dialog or there was a problem displaying it.
	;	Currently assuming the user canceled, otherwise this would tell us whether an error
	;	occurred vs. the user canceling: if (result != HRESULT_FROM_WIN32(ERROR_CANCELLED))
	ObjRelease(pFD)
	return vRet
}

;==================================================

BIFEx_StrStarts(vText, vNeedle, vCaseSen:=0)
{
	if !vCaseSen
		return (SubStr(vText, 1, StrLen(vNeedle)) = vNeedle) ;case insensitive
	else
		return (SubStr(vText, 1, StrLen(vNeedle)) == vNeedle)
}

;==================================================

BIFEx_StrEnds(vText, vNeedle, vCaseSen:=0)
{
	if !vCaseSen
		return (SubStr(vText, StrLen(vText)+1-StrLen(vNeedle)) = vNeedle) ;case insensitive
	else
		return (SubStr(vText, StrLen(vText)+1-StrLen(vNeedle)) == vNeedle)
}

;==================================================

BIFEx_InStrRev(vText, vNeedle, vCaseSen:=0)
{
	static vIsV1 := InStr(10, 1,, -1, 1)
	static vPos := vIsV1 ? 0 : -1
	return InStr(vText, vNeedle, vCaseSen, vPos)
}

;==================================================

BIFEx_Swap(ByRef vVar1, ByRef vVar2)
{
	local vTemp
	vTemp := vVar1
	vVar1 := vVar2
	vVar2 := vTemp
}

;==================================================

BIFEx_ArrSort(oArray, vOpt:="", vFunc:="")
{
	;[FIXME] note: vOpt is currently unused
	local oArray2, oBuf, vCount, vNum
	static vIsV1 := (InStr(A_AhkVersion, "1.") == 1)
	static vThisFunc := "BIFEx_ArrSort"
	static pFunc := RegisterCallback(vThisFunc "_Cmp", "C", 2)
	;static pFunc := CallbackCreate(vThisFunc "_Cmp", "C", 2)
	if IsObject(vFunc) ;if function object
		%A_ThisFunc%_Cmp(3, vFunc)
	else if (vFunc == "") ;if omitted
		%A_ThisFunc%_Cmp(3, 0)
	else ;if function name
	{
		if !(oFunc := Func(vFunc))
			throw Exception("", -1)
		%A_ThisFunc%_Cmp(3, oFunc)
	}
	vCount := oArray.Length()
	oBuf := Buffer(vCount*8)
	Loop % vCount
		NumPut(A_Index, oBuf.Ptr, A_Index*8-8, "Int64")
	%A_ThisFunc%_Cmp(1, oArray)
	DllCall("msvcrt\qsort", "Ptr",oBuf.Ptr, "UPtr",vCount, "UPtr",8, "Ptr",pFunc, "Cdecl")
	%A_ThisFunc%_Cmp(2, 0)
	oOutput := []
	(vIsV1) ? oOutput.SetCapacity(vCount) : oOutput.Capacity := vCount
	Loop % vCount
	{
		vNum := NumGet(oBuf.Ptr, A_Index*8-8, "Int64")
		oOutput.Push(oArray[vNum])
	}
	return oOutput
}

;==================================================

BIFEx_ArrSort_Cmp(pIndex1, pIndex2, vOffset:=0)
{
	local vIndex1, vIndex2, vValue1, vValue2
	static oArray, oFunc
	if (pIndex1 == 1)
	{
		oArray := pIndex2
		return
	}
	else if (pIndex1 == 2)
	{
		oArray := oFunc := ""
		return
	}
	else if (pIndex1 == 3)
	{
		oFunc := pIndex2
		return
	}
	vIndex1 := NumGet(pIndex1+0, 0, "Int64")
	vIndex2 := NumGet(pIndex2+0, 0, "Int64")
	vValue1 := oArray[vIndex1]
	vValue2 := oArray[vIndex2]
	vOffset := vIndex2 - vIndex1
	if IsObject(oFunc)
		return oFunc.Call(vValue1, vValue2, vOffset)
	return (vValue1 > vValue2) ? 1 : (vValue1 < vValue2) ? -1 : -vOffset
}

;==================================================

AHKFC_HotIfWin(vFuncName, oParams*)
{
	vFuncName := SubStr(vFuncName, 4)

	if oParams.HasKey(1)
	{
		if IsInteger(oParams[1])
		&& (Type(oParams[1]) == "Integer")
			oParams[1] := "ahk_id " oParams[1]
		else if IsObject(oParams[1])
		{
			if !oParams[1].Hwnd
				throw Exception("", -2)
			oParams[1] := "ahk_id " oParams[1].Hwnd
		}
	}

	if !oParams.Length()
		Hotkey, % "If"
	else if oParams.HasKey(1) && !oParams.HasKey(2)
		Hotkey, % vFuncName, % oParams[1]
	else if oParams.HasKey(1)
		Hotkey, % vFuncName, % oParams[1], % oParams[2]
	else if oParams.HasKey(2)
		Hotkey, % vFuncName,, % oParams[2]
}

;==================================================

BIF_ControlGetPos(ByRef vCtlX, ByRef vCtlY, ByRef vCtlW, ByRef vCtlH, hCtl)
{
	local hWndParent, oRECT
	;hWndParent := DllCall("user32\GetAncestor", "Ptr",hCtl, "UInt",1, "Ptr") ;GA_PARENT := 1
	hWndParent := AHKFC_GetNonChildParent(hCtl)
	oRECT := Buffer(16, 0)
	DllCall("user32\GetWindowRect", "Ptr",hCtl, "Ptr",oRECT.Ptr)
	DllCall("user32\MapWindowPoints", "Ptr",0, "Ptr",hWndParent, "Ptr",oRECT.Ptr, "UInt",2)
	vCtlX := NumGet(oRECT.Ptr, 0, "Int")
	vCtlY := NumGet(oRECT.Ptr, 4, "Int")
	vCtlW := NumGet(oRECT.Ptr, 8, "Int") - vCtlX
	vCtlH := NumGet(oRECT.Ptr, 12, "Int") - vCtlY
}

;==================================================

BIF_ControlMove(vCtlX:="", vCtlY:="", vCtlW:="", vCtlH:="", hCtl:="")
{
	local hWnd, hWndParent, oRECT
	if !hCtl
		return

	hWnd := AHKFC_GetNonChildParent(hCtl)
	hWndParent := DllCall("user32\GetAncestor", "Ptr",hCtl, "UInt",1, "Ptr") ;GA_PARENT := 1
	oRECT := Buffer(16, 0)
	DllCall("user32\GetWindowRect", "Ptr",hCtl, "Ptr",oRECT.Ptr)
	DllCall("user32\MapWindowPoints", "Ptr",0, "Ptr",hWnd, "Ptr",oRECT.Ptr, "UInt",2)
	if (vCtlX == "")
		vCtlX := NumGet(oRECT.Ptr, 0, "Int")
	if (vCtlY == "")
		vCtlY := NumGet(oRECT.Ptr, 4, "Int")
	if (vCtlW == "")
		vCtlW := NumGet(oRECT.Ptr, 8, "Int") - NumGet(oRECT.Ptr, 0, "Int")
	if (vCtlH == "")
		vCtlH := NumGet(oRECT.Ptr, 12, "Int") - NumGet(oRECT.Ptr, 4, "Int")

	;MoveWindow function (winuser.h) | Microsoft Docs
	;https://docs.microsoft.com/en-us/windows/win32/api/winuser/nf-winuser-movewindow
	;For a child window, they are relative to the upper-left corner of the parent window's client area.

	;AHK source code: 'MoveWindow accepts coordinates relative to the control's immediate parent'
	if (hWndParent != hWnd)
		DllCall("user32\MapWindowPoints", "Ptr",hWnd, "Ptr",hWndParent, "Ptr",oRECT.Ptr, "UInt",1)
	DllCall("user32\MoveWindow", "Ptr",hCtl, "Int",vCtlX, "Int",vCtlY, "Int",vCtlW, "Int",vCtlH, "Int",1)
}

;==================================================

AHKFC_RandomInt64()
{
	local Rand1, Rand2
	Random, Rand1, -0x80000000, 0x7FFFFFFF
	Random, Rand2, -0x80000000, 0x7FFFFFFF
	Rand1 += 0x80000000
	Rand2 += 0x80000000
	return (Rand1 << 32) | Rand2
}

;==================================================

;where vNum1/vNum2/return value are UInt64s stored as Int64
BIFEx_ModUInt64(vNum1, vNum2)
{
	local vRem, vRem1, vRem2
	;note: 'big' numbers (the top half of UInt64) as Int64, will be negative
	;WARNING: keep '-0x8000000000000000' intact, e.g. '- 0x8000000000000000' will silently give an incorrect value in AHK v1
	if !vNum2
		throw Exception("Divide by zero.", -1)
	;else if (vNum2 == 1)
	;	return 0
	;else if (vNum2 == 2)
	;	return vNum1 & 1
	else if (vNum1 >= 0)
	{
		if (vNum2 >= 0) ;small vNum1, small vNum2
			;return vNum1 - (vNum1//vNum2)*vNum2
			return Mod(vNum1, vNum2)
		else ;small vNum1, big vNum2
			return vNum1
	}
	else
	{
		if (vNum2 >= 0) ;big vNum1, small vNum2
		{
			;max vNum1 is 0xFFFFFFFFFFFFFFFF (stored as Int64: -1)
			;min vNum1 is 0x8000000000000000 (stored as Int64: -0x8000000000000000)
			;max vNum2 is 0x7FFFFFFFFFFFFFFF (stored as Int64: the same)
			;min vNum2 is 1

			;we calculate the remainder in 2 parts:
			;e.g. vRem1 = Mod(0x8000000000000000, vNum2) = -Mod(-0x8000000000000000, -vNum2)
			;e.g. vRem2 = Mod(vNum1+(-0x8000000000000000), vNum2)

			;max vRem1 is -Mod(-0x8000000000000000, -0x4000000000000001) = 0x3FFFFFFFFFFFFFFF
			;max vRem2 is Mod(0x7FFFFFFFFFFFFFFF, 0x4000000000000000) = 0x3FFFFFFFFFFFFFFF
			;for reference: Mod(0x7FFFFFFFFFFFFFFF, 0x4000000000000001) = 0x3FFFFFFFFFFFFFFE
			;so max vRem1+vRem2 is Int64

			;to find a max value you want to subtract a number (zero times or) only once, and subtract the smallest possible number:
			;e.g. vRem1 = -Mod(-0x8000000000000000, -0x4000000000000001)

			vRem1 := -Mod(-0x8000000000000000, -vNum2)
			vRem2 := Mod(-0x8000000000000000+vNum1, vNum2)
			vRem := vRem1 + vRem2

			;vRem must end up min 0, and max vNum2-1:
			;note: vRem1 < vNum2
			;note: vRem2 < vNum2
			;so: vRem1+vRem2 = vRem < vNum2*2
			;so: vRem - vNum2 < vNum2
			if (vRem >= vNum2) ;an example where vRem1+vRem2 = vNum2: 0x8000000000000002 % 10 = 0, where vRem1+vRem2 = 8+2 = 10
				vRem -= vNum2
			return vRem
		}
		else ;big vNum1, big vNum2
			return (vNum1 >= vNum2) ? (vNum1-vNum2) : vNum1
		;e.g. UInt64s as Int64s, converted to UInt64s, then mod applied:
		;e.g. -0x8000000000000000 % -1 is 0x8000000000000000 % 0xFFFFFFFFFFFFFFFF = 0x8000000000000000
		;e.g. -1 % -0x8000000000000000 is 0xFFFFFFFFFFFFFFFF % 0x8000000000000000 = 0xFFFFFFFFFFFFFFFF - 0x8000000000000000 = 0x7FFFFFFFFFFFFFFF
	}
}

;==================================================

BIFEx_BitShiftRightLogical(vNum, vShift)
{
	;local
	if (vShift == 0)
		return vNum
	else if (vShift < 0) || (vShift > 63)
		throw Exception("Error evaluating expression.", -1)
	return (vNum >> vShift) & ((1 << (64-vShift))-1)
	;return (vNum >> vShift) & ((2**(64-vShift))-1) ;equivalent to line above
}

;==================================================
